package tools3d;

import generaltools.Randomizer;
import freeware.PrintfFormat; // used in CDK package 
import java.util.Arrays;
import java.util.Random;
import Jama.*;
import numerictools.DoubleArrayTools;

/** Mostly static helper methods for Vector3D class */
public class Vector3DTools {

    public static final String HASH_DELIM = "_";

    private static Random rnd = Randomizer.getInstance();

    /** generates vector according to polar coordinates phi, theta, r */
    public static Vector3D generateCartesianFromPolar(double phi, double theta, double r) {
	return new Vector3D(r*Math.cos(phi)*Math.sin(theta),
			    r*Math.sin(phi)*Math.sin(theta),
			    r*Math.cos(theta));
    }

    /** generates vector of length 1 with randomized direction */
    public static Vector3D generateRandomDirection() {
	assert rnd != null;
	double phi = 2.0 * Math.PI * rnd.nextDouble();
	double theta = Math.PI * rnd.nextDouble();
	Vector3D result = generateCartesianFromPolar(phi, theta, 1.0);
	assert result.isValid();
	return result;
    }

    /** generates vector of that is orthogonal to v, using parts of p */
    public static Vector3D generateOrthogonalDirection(Vector3D vOrig, Vector3D pOrig) {
	assert (vOrig.lengthSquare() > 0.0);
	assert (pOrig.lengthSquare() > 0.0);
	Vector3D v = new Vector3D(vOrig);
	Vector3D p = new Vector3D(pOrig);
	v.normalize();
	p.normalize();
	Vector3D result = p.minus(v.mul(p.dot(v)));
	assert (result.lengthSquare() > 0.0);
	result.normalize();
	assert (Math.abs(result.dot(vOrig)) < 0.01); // check if really orthogonal
	return result;
    }

    /** generates vector of length 1 with randomized direction */
    public static Vector3D generateRandomOrthogonalDirection(Vector3D vOrig) {
	assert (vOrig.lengthSquare() > 0.0);
	Vector3D v = new Vector3D(vOrig);
	v.normalize();
	Vector3D p = null;
	do {
	    p = generateRandomDirection();
	} while ( (p.cross(v)).lengthSquare() == 0.0 ); // try again if parallel
	return generateOrthogonalDirection(v, p);
    }

    /** generate set of n points with radius r around origin vector */
    public static Vector3D[] generateSpherePoints(int n, 
				     double r, Vector3D origin) {
	Vector3D[] ray = new Vector3D[n];
	if (n == 0) {
	    return ray;
	}
	double h,theta,phi;
	double oldPhi = 0.0;
	for (int i = 1; i < n-1; ++i) {
	    ray[i] = new Vector3D();
	    h = -1.0 + 2.0 * i / (n-1.0);
	    // ASSERT((h>=-1.0)&&(h<=1.0), exception);
	    theta = Math.acos(h);
	    phi = (oldPhi + 3.6/Math.sqrt(n*(1-h*h)));
	    ray[i].setX(Math.cos(phi) * Math.sin(theta));
	    ray[i].setY(Math.sin(phi)*Math.sin(theta));
	    ray[i].setZ(Math.cos(theta));
	    oldPhi = phi;
	}
	ray[0] = new Vector3D(0.0,0.0,-1.0);
	ray[n-1] = new Vector3D(0.0,0.0,1.0);
	for (int i = 0; i < ray.length; ++i) {
	    ray[i].scale(r);
	    ray[i].add(origin);
	    // ray[i] = (ray[i] * r) + origin;
	}
	return ray;
    }

    /** computes distance root mean square */
    public static double computeDrms(Vector3D[] v1, Vector3D[] v2) {
	double result = 0.0;
	double term = 0.0;
	for (int i = 0; i < v1.length; ++i) {
	    for (int j = i+1; j < v1.length; ++j) {
		term = (v1[i].minus(v1[j])).length()-(v2[i].minus(v2[j])).length();
		result += (term * term);
	    }
	}
	double norm = 0.5 * v1.length * (v1.length-1);
	result /= norm;
	result = Math.sqrt(result);
	return result;
    }

    /** computes root mean square between two vector array, NOT attempting to superpose them */
    public static double computeRms(Vector3D[] v1, Vector3D[] v2) {
	double result = 0.0;
	double term = 0.0;
	for (int i = 0; i < v1.length; ++i) {
	    result += v1[i].distanceSquare(v2[i]);
	}
	result /= v1.length;
	result = Math.sqrt(result);
	return result;
    }

    /** computes center of gravity of vector set */
    public static Vector3D computeCenterGravity(Vector3D[] v) {
	assert v.length > 0;
	Vector3D sum = new Vector3D(0,0,0);
	for (int i = 0; i < v.length; ++i) {
	    sum.add(v[i]);
	}
	sum.scale(1.0/v.length);
	return sum;
    }

    /** computes sorted array of all vector-vector angles */
    public static double[] computeVectorFieldSignature(Vector3D[] directions) {
	int n = directions.length;
	int n2 = (n * (n - 1)) / 2; 
	double[] result = new double[n2];
	int pc = 0;
	for (int i = 0; i < n; ++i) {
	    for (int j = 0; j < i; ++j) {
		result[pc++] = directions[i].angle(directions[j]);
	    }
	}
	Arrays.sort(result); // sort in ascending order
	assert(pc == result.length);
	return result;
    }

    /** computes score indicating how different vector fields are. Score zero corresponds to perfect match */
    public static double computeVectorFieldSimilarityScore(Vector3D[] directions1, Vector3D[] directions2) {
	assert (directions1.length > 1);
	assert (directions1.length == directions2.length);
	double[] sig1 = computeVectorFieldSignature(directions1);
	double[] sig2 = computeVectorFieldSignature(directions2);
	double maxDiff = Math.abs(sig1[0]-sig2[0]);
	for (int i = 1; i < sig1.length; ++i) {
	    double diff = Math.abs(sig1[i]-sig2[i]);
	    if (diff > maxDiff) {
		maxDiff = diff;
	    }
	}
	assert (maxDiff >= 0.0);
	return maxDiff;
    }

    public static String prettyString(Vector3D vector, int len, int digits) {
	assert digits >= 0;
	PrintfFormat positionFormat = new PrintfFormat("%" + len + "." + digits + "f");
	return "" + positionFormat.sprintf(vector.getX()) + " "
	    + positionFormat.sprintf(vector.getY()) + " "
	    + positionFormat.sprintf(vector.getZ());
    }

    private static void updateMomentOfInertiaMatrix(Matrix3D m, Vector3D v, double mass, Vector3D rotationCenter) {
	Vector3D d = v.minus(rotationCenter);
	double xx = mass * d.getX() * d.getX();
	double xy = mass * d.getX() * d.getY();
	double xz = mass * d.getX() * d.getZ();
	double yx = mass * d.getY() * d.getX();
	double yy = mass * d.getY() * d.getY();
	double yz = mass * d.getY() * d.getZ();
	double zx = mass * d.getZ() * d.getX();
	double zy = mass * d.getZ() * d.getY();
	double zz = mass * d.getZ() * d.getZ();
	m.setXX(m.getXX()+xx);
	m.setXY(m.getXY()+xy);
	m.setXZ(m.getXZ()+xz);
	m.setYX(m.getYX()+yx);
	m.setYY(m.getYY()+yy);
	m.setYZ(m.getYZ()+yz);
	m.setZX(m.getZX()+zx);
	m.setZY(m.getZY()+zy);
	m.setZZ(m.getZZ()+zz);
    }

    public static Matrix3D getMomentOfInertiaMatrix(Vector3D[] v, double[] masses, Vector3D rotationCenter) {
	assert v.length == masses.length;
	Matrix3D result = new Matrix3D(0.0);
	for (int i = 0; i < v.length; ++i) {
	    updateMomentOfInertiaMatrix(result, v[i], masses[i], rotationCenter);
	}
	return result;
    }

    public static Matrix3D getMomentOfInertiaMatrix(Vector3D[] v, double[] masses) {
	Vector3D rotationCenter = computeCenterGravity(v);
	return getMomentOfInertiaMatrix(v, masses, rotationCenter);
    }

    /** returns radii of gyration of body */
    public static double[] getRadiiOfGyration(Vector3D[] v, double[] masses) {
	assert v.length > 0;
	assert v.length == masses.length;
	Matrix3D inertiaMatrix = getMomentOfInertiaMatrix(v, masses);
	Matrix matrix = new Matrix(inertiaMatrix.toArray()); // generate JAMA matrix
	EigenvalueDecomposition decomp = new EigenvalueDecomposition(matrix);
	double[] eigenValues = decomp.getRealEigenvalues();
	assert eigenValues.length == 3;
	double totalMass = DoubleArrayTools.computeSum(masses);
	assert totalMass > 0.0;
	for (int i = 0; i < eigenValues.length; ++i) {
	    eigenValues[i] = Math.sqrt(eigenValues[i]/totalMass);
	}
	// sort eigenvalues, such that highest eigenvalue is first:
	Arrays.sort(eigenValues);
	double tmp = eigenValues[0]; // swap elements 0 and 2:
	eigenValues[0] = eigenValues[2];
	eigenValues[2] = tmp;
	return eigenValues;
    }

    public static String computeEpsilonHash(double x, double epsilon) {
	return "" + Math.round(x / epsilon);
    }

    public static String computeEpsilonHash(Vector3D v, double epsilon) {
	return computeEpsilonHash(v.getX(), epsilon) + HASH_DELIM
	    + computeEpsilonHash(v.getY(), epsilon) + HASH_DELIM
	    + computeEpsilonHash(v.getZ(), epsilon);
    }

    public static String computeEpsilonHash(Vector4D v, double epsilon) {
	return computeEpsilonHash(v.getX(), epsilon) + HASH_DELIM
	    + computeEpsilonHash(v.getY(), epsilon) + HASH_DELIM
	    + computeEpsilonHash(v.getZ(), epsilon);
    }

}
