library(bio3d)

# R script computes theoretical hydrodynamic radius for a PDB file of an RNA structure.
# see: http://en.wikipedia.org/wiki/Hydrodynamic_radius
# Usage: Rscript trhydro.R PDBFILE
# or R --vanilla --args PDBFILE < trhydro.R
# with PDBFILE being the filename of the PDB structure. The script has in this example to be in the 
# same directory as the PDB file.
# Eckart Bindewald, 2010

helpOutput <- function() {
  cat("Program computes the maximum distance between any two residues of an RNA structure. Each residue position is represented through its phosphate atom position. Input is a PDB file. This corresponds to the minimum radius of a sphere containing the structure.\n")
}

dist.euclid <- function(x1,y1,z1,x2,y2,z2) {
  sqrt((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) + (z1-z2)*(z1-z2) )
}

dist.euclidSq <- function(x1,y1,z1,x2,y2,z2) {
  (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2) + (z1-z2)*(z1-z2)
}

dist.euclidSqD <- function(x1,x2,x3) {
 x1*x1 + x2*x2 + x3*x3
}

test.dist.euclid <- function() {
   print(dist.euclid(3,0,0,0,4,0))
}

radius.max <- function(pdb) {
  p.inds <- atom.select(pdb, "///////")
#  n <- length(p.inds)
  xyz <- pdb$xyz[ p.inds$xyz ]
  n3 <- length(xyz)
  xv <- xyz[seq(1,n3,3)]
  yv <- xyz[seq(2,n3,3)]
  zv <- xyz[seq(3,n3,3)]
  n <- length(xv)
  stopifnot((n * 3) == n3)
  rMax2 <- 0
  xsum <- 0
  ysum <- 0
  zsum <- 0
  for (i in 1:n) {
    xsum <- xsum + xv[i]
    ysum <- ysum + yv[i]
    zsum <- zsum + zv[i]
  }
  xc <- xsum / n
  yc <- ysum / n
  zc <- zsum / n  # coordinates of coordinate center 
  for (i in 1:n) {
        dx <- xc - xv[i]
        dy <- yc - yv[i]
        dz <- zc - zv[i]
        r2 <- dx*dx + dy*dy + dz*dz
        if (r2 > rMax2) {
            rMax2 <- r2
        } 
  }
  result <- sqrt(rMax2) 
  result
}

## main program

args <- commandArgs(trailingOnly=TRUE)
if (length(args) < 1) {
	helpOutput()
} else {
 filename <- args[1]
 pdb <- read.pdb(filename)
 rMax <- radius.max(pdb)
 cat("Sphere radius:", rMax, "Angstroem\n")
}
