#ifndef _SORTED_COMPRESSED_SEQUENCE_
#define _SORTED_COMPRESSED_SEQUENCE_

#include <iostream>
#include <algorithm>
#include <Vec.h>
#include <debug.h>
#include <Random.h>
#include <Timer.h>
#include <limits.h>

using namespace std;

#define CHAR_D ( CHAR_MAX - CHAR_MIN )
// #define CHAR_D2 ( CHAR_D * CHAR_D )
// #define CHAR_D3 ( CHAR_D * CHAR_D * CHAR_D )

class SortedCompressedSequenceConstIterator;

class SortedCompressedSequence {

 public:

  enum { STORE_INTERVALL = 50 }; // not more than this many differences but store absolute value in hash

  typedef SortedCompressedSequenceConstIterator const_iterator;

  typedef SortedCompressedSequenceConstIterator iterator;

  typedef string compressed_type;

  typedef int value_type;

  typedef Vec<int> set_type;

  typedef compressed_type::size_type size_type;

  typedef compressed_type::difference_type difference_type;

  typedef map<size_type, value_type> map_type; // used to store too large values

 private:

  compressed_type s; // this string encode numeric data!

  map_type additional;

 public:

  /** Default constructor */
  SortedCompressedSequence() { } 

  /** Construction using un-compressed set */
  SortedCompressedSequence(const set_type& content) {
    add(content);
  }

  /** Copy constructor */
  SortedCompressedSequence(const SortedCompressedSequence& other) : s(other.s), additional(other.additional) {
  }

  /** Virtual destructor */
  virtual ~SortedCompressedSequence() { }

  /** Assignment operator */
  virtual SortedCompressedSequence& operator = (const SortedCompressedSequence& other) {
    if (this != &other) {
      copy(other);
    }
    return *this;
  }

  /** Concatenates content to end */
  virtual void add(const set_type& content);
  
  /** Returns iterator to beginning of container */
  virtual SortedCompressedSequenceConstIterator begin() const;
  
  /** Removes all content */
  virtual void clear() {
    s = "";
    additional.clear();
  }

  /** Copy method , being used by copy constructor and assignment operator */
  virtual void copy(const  SortedCompressedSequence& other) {
    s = other.s;
    additional = other.additional; 
  }

  /** Returns iterator to end of container */ 
  virtual SortedCompressedSequenceConstIterator end() const;

  /** Returns internal representation */
  virtual const compressed_type internal() const { return s; }

  /** Returns number of stored items */
  virtual size_type size() const { return s.size(); }

  virtual set_type toVector() const;

  friend class SortedCompressedSequenceConstIterator;

};

inline
void SortedCompressedSequence::add(const set_type& set) {
  // cout << "Starting add " << set << endl;
  if (set.size() == 0) {
    return; // do nothing
  }
  string::size_type resultLen = static_cast<string::size_type>(set.size());  // plus 1: separater for size string
  string result(resultLen,' ');
  for (set_type::size_type i = 0; i < set.size(); ++i) {
    value_type diff = set[i];
    if (i > 0) {
      diff -= set[i-1];
      ASSERT(diff > 0);
    }
    if ((diff <= CHAR_D) && (i > 0) && ((i%STORE_INTERVALL) != 0))  {
      ASSERT(i < result.size());
      result[i] = diff + CHAR_MIN;
    } else {
      result[i] = CHAR_MIN;
      additional[i] = set[i];
    }
  }
  s = s + result;
}

/** Iterator for compressed class. Trying: random_access_iterator: so far too slow because of linear inc() operator */
class SortedCompressedSequenceConstIterator : public std::iterator<std::random_access_iterator_tag,
					      SortedCompressedSequence::value_type>{

 public:

  typedef string _Alloc;

  typedef SortedCompressedSequence::compressed_type compressed_type; 

  typedef SortedCompressedSequence::set_type set_type; 

  typedef SortedCompressedSequence::size_type size_type; 

  typedef SortedCompressedSequence::value_type value_type;

  typedef SortedCompressedSequence vector_type;

  typedef compressed_type::difference_type difference_type;

  typedef int ptrdiff_t;

 private:

  const vector_type *container;

  size_type lastPos; // cached last position

  value_type lastVal; // cached last value

  size_type pos; // current internal position on string 

  value_type val; // current content value

 public:

  /** Standard constructor */
  SortedCompressedSequenceConstIterator(const vector_type *_container) : container(_container), lastPos(0), lastVal(0), pos(0), val(0) {
    interpretPosition(); // sets value and next jump
  }

  /** Default constructor */
  SortedCompressedSequenceConstIterator() : container(0), lastPos(0), lastVal(0), pos(0), val(0) {
  }

  /** Constructor using an internal position. Should be private FIXIT */
  SortedCompressedSequenceConstIterator(const vector_type *_container, size_type _pos) : container(_container), 
    lastPos(0), lastVal(0), pos(_pos), val(0) {
    interpretPosition(); // sets value and next jump
  }

  /** Copy constructor */
  SortedCompressedSequenceConstIterator(const SortedCompressedSequenceConstIterator& other) 
    : container(other.container), lastPos(other.lastPos), lastVal(other.lastVal), pos(other.pos), val(other.val) { }
  
  /** Virtual destructor */
  virtual ~SortedCompressedSequenceConstIterator() { }

  /** Assignment operator */
  SortedCompressedSequenceConstIterator& operator = (const SortedCompressedSequenceConstIterator& other) {
    if (&other != this) {
      copy(other);
    }
    return *this;
  }

  /** Copy method */
  virtual void copy(const SortedCompressedSequenceConstIterator& other) {
    container = other.container;
    lastPos = other.lastPos;
    lastVal = other.lastVal;
    pos = other.pos;
    val = other.val;
  }

  /** Returns position iterator is pointing to */ 
  virtual size_type getPos() const { return pos; }

  /** Increment */
  virtual void inc() {
    ASSERT(*this != container->end());
    lastPos = pos;
    lastVal = val;
    ++pos;
    interpretPosition(); 
  }

  /** Increment */
  virtual void dec() {
    ASSERT(*this != container->begin());
    lastPos = pos;
    lastVal = val;
    --pos;
    interpretPosition(); 
  }

  /** Access to element that iterator is pointing to */
  value_type operator * ( ) {
    return val;
  }

  /** prefix ++ operator */
  void operator ++ () {
    ASSERT(*this != container->end());
    const compressed_type& set = container->s;
    // lastPos = pos;
    // lastVal = val;
    ++pos;
    // if (pos < set.size()) {
    if (set[pos] > CHAR_MIN) {
      val += (set[pos] - CHAR_MIN);
    } else {
      val = (container->additional).find(pos)->second;
    }
    // } else {
    //  pos = set.size(); // value undefined
    // }
  }

  /** postfix ++ operator.  "int" indicates postfix, bizarre C++ idiom... */
  void operator ++ (int) {
    ASSERT(*this != container->end());
    const compressed_type& set = container->s;
    // ASSERT(pos < set.size());
    // lastPos = pos;
    // lastVal = val;
    ++pos;
    // if (pos < set.size()) {
    if (set[pos] > CHAR_MIN) {
      val += (set[pos] - CHAR_MIN);
    } else {
      val = (container->additional).find(pos)->second;
    }
      // } else {
      // pos = set.size(); // value undefined
      // }
  }

  /** prefix -- operator */
  void operator -- () {
    ASSERT(*this != container->begin());
    dec();
  }
  
  /** postfix -- operator */
  void operator -- (int) {
    ASSERT(*this != container->begin());
    dec();
  }
  
  /** += operator. */
  void operator += (size_type n) {
    ASSERT(*this != container->end());
    lastPos = pos;
    lastVal = val;
    pos += n;
    interpretPosition();
  }
  
  /** -= operator. */
  void operator -= (size_type n) {
    ASSERT(*this != container->begin());
    lastPos = pos;
    lastVal = val;
    pos -= n;
    interpretPosition();
  }
  
  /** Comparison operator */
  bool operator == (const SortedCompressedSequenceConstIterator& other) const {
    return (container == other.container) && (pos == other.pos);
  }
  
  /** Negated equality */
  bool operator != (const SortedCompressedSequenceConstIterator& other) const {
    return !(*this == other);
  }
  
					      private:
  
  /** Interprets compressed string at current position and set val member */
  void interpretPosition();
  
};
  
  /** Interprets compressed string at current position and sets val members */
inline
void
SortedCompressedSequenceConstIterator::interpretPosition() {
  // cout << "Starting interPretPosition with " << pos << " " << val << endl;
  const compressed_type& set = container->s;
  val = 0;
  if ((set.size() == 0) || (pos >= set.size())) {
    return;
  }
  int lastPosi = static_cast<int>(lastPos);
  for (int i = static_cast<int>(pos); i >= 0; i--) {
    if ((i == lastPosi) && (lastPosi > 0)) {
      val += lastVal;
      break;
    }
    if (set[i] > CHAR_MIN) {
      val += (set[i] - CHAR_MIN);
    } else {
      // ASSERT((container->additional).find(i) != (container->additional).end());
      val += (container->additional).find(i)->second;
      break;
    }
  }
}

inline
SortedCompressedSequenceConstIterator
SortedCompressedSequence::begin() const { return SortedCompressedSequenceConstIterator(this, 0); }
  
inline
SortedCompressedSequenceConstIterator
SortedCompressedSequence::end() const { return SortedCompressedSequenceConstIterator(this, s.size()); }

inline
SortedCompressedSequence::set_type
SortedCompressedSequence::toVector() const {
  // cout << "Started SortedCompressedSequence.toVector(): " << size() << endl;
  if (size() == 0) {
    return set_type();
  }
  set_type result;// (size());
  result.reserve(size());
  for (const_iterator it = begin(); it != end(); it++) {
    result.push_back(*it);
  } 
  ASSERT(result.size() == size());
  // cout << "Finished SortedCompressedSequence.toVector()..." << endl;
  return result;
}

inline
ostream& 
operator << (ostream& os, const SortedCompressedSequence& seq) {
  os << seq.size() << "  ";
  SortedCompressedSequence::size_type count = 0;
  for (SortedCompressedSequence::const_iterator it = seq.begin(); it != seq.end(); it++) {
    os << (*it) << " ";
    ++count;
  }
  //  ASSERT(count == seq.size()); // must be this many entries
  return os;
}

inline
SortedCompressedSequenceConstIterator::ptrdiff_t 
operator - (const SortedCompressedSequenceConstIterator& left, const SortedCompressedSequenceConstIterator& right) {
  return (static_cast<ptrdiff_t>(left.getPos()) - static_cast<ptrdiff_t>(right.getPos()));
}
     
/** Tests compression and uncompression */
class SortedCompressedSequenceTest {

 public:
  
  typedef SortedCompressedSequence::value_type value_type;
  typedef SortedCompressedSequence::set_type set_type;
  typedef SortedCompressedSequence::size_type size_type;

  static void testASet(const set_type& testSet) {
    cout << "Starting SortedCompressedSequence::testASet" << endl;
    cout << "Original set: " << testSet << endl;
    SortedCompressedSequence compressed(testSet);
    ASSERT(compressed.size() == testSet.size());
    //    compressed_type compressed = compressSet(testSet);
    cout << "Compressed version: " << compressed.size() << " " << compressed  << endl;
    SortedCompressedSequence::compressed_type internal = compressed.internal(); // internal representation
    cout << "Internal representation: " << internal << endl;
    for (SortedCompressedSequence::compressed_type::const_iterator it = internal.begin(); it != internal.end(); it++) {
      cout << static_cast<int>(*it) << " " << (static_cast<int>(*it) - CHAR_MIN) << " " << (*it) << endl;
    }
    set_type uncompressed = compressed.toVector();
    cout << "Uncompressed set: " << uncompressed << endl;
    ASSERT(testSet == uncompressed);
    cout << "Testing iterator: " << endl;
    size_type count = 0;
    for (SortedCompressedSequence::const_iterator it = compressed.begin(); it != compressed.end(); it++) {
      cout << ++count << " " << (*it) << endl;
    }
    ASSERT(count == testSet.size());

    cout << "Finished SortedCompressedSequence::testASet" << endl;

  }

  static void test1() {
    set_type testSet;
    testSet.push_back(2);
    testSet.push_back(4);
    testSet.push_back(8);
    testSet.push_back(16);
    testASet(testSet);
  }

  static void test2() {
    set_type testSet;
    value_type val = 1;
    for (size_type i = 1; i < CHAR_D; i *= 2) {
      testSet.push_back(val);
      val += i;
    }
    testASet(testSet);
  }

  static void test3() {
    set_type testSet;
    value_type val = 1;
    for (size_type i = 1; i < (CHAR_D*CHAR_D); i *= 2) {
      testSet.push_back(val);
      val += i;
    }
    testASet(testSet);
  }

  static void test4() {
    set_type testSet;
    value_type val = 1;
    for (size_type i = 1; i < (CHAR_D*CHAR_D*CHAR_D); i *= 2) {
      testSet.push_back(val);
      val += i;
    }
    testASet(testSet);
  }

  static set_type generateRandomSortedVec(value_type min, value_type max, size_type count) {
    set_type result;
    result.reserve(count);
    Random rnd = Random::getInstance();
    value_type diff = max - min;
    result.push_back(rnd.getRand(diff) + min);
    for (size_type i = 0; i < count; ++i) {
      result.push_back(result[result.size()-1] + rnd.getRand(diff) + 1);
    }
    return result;
  }

  /** Tests, if set_intersection algorithm works */
  static void testIntersection() {
    cout << "Starting testIntersection..." << endl;
    set_type set1 = generateRandomSortedVec(1,300, 10000);
    set_type set2 = generateRandomSortedVec(1,300, 10000);
    cout << "Testing intersection between sets " << endl << set1.size() << endl << "and" << endl << set2.size() << endl;
    set_type set3;
    set_type cset3;
    Timer timer1, timer2;
    timer1.start();
    for (int i = 0; i < 200; i++) {
      set3.clear();
      set_intersection(set1.begin(), set1.end(), set2.begin(), set2.end(), back_inserter(set3));
    }
    timer1.stop();
    cout << "intersection of vector set " << set3 << endl;
    cout << "Timer: " << timer1 << endl;
    SortedCompressedSequence cset1(set1);
    SortedCompressedSequence cset2(set2);
    timer2.start();
    for (int i = 0; i < 200; i++) {
      cset3.clear();
      set_intersection(cset1.begin(), cset1.end(), cset2.begin(), cset2.end(), back_inserter(cset3));
    }
    timer2.stop();
    cout << "intersection of compressed set " << cset3 << endl;
    cout << "Timer: " << timer2 << endl;
    ASSERT(cset3 == set3);
    cout << "Finished testIntersection..." << endl;
 }

  /** Tests, if set_intersection algorithm works */
  static void testLowerBound() {
    cout << "Starting testIntersection..." << endl;
    set_type set1 = generateRandomSortedVec(1,300, 10000);
    set_type set2 = generateRandomSortedVec(1,300, 10000);
    cout << "Testing lower bound between sets " << endl << set1.size() << endl << "and" << endl << set2.size() << endl;
    set_type::iterator lb;
    Timer timer1, timer2;
    timer1.start();
    for (int i = 0; i < 200; i++) {
      lb = lower_bound(set1.begin(), set1.end(), *(set2.begin()));
    }
    timer1.stop();
    cout << "Lower bound of vector set " << (*lb) << endl;
    cout << "Timer: " << timer1 << endl;
    SortedCompressedSequence cset1(set1);
    SortedCompressedSequence cset2(set2);
    SortedCompressedSequence::iterator lb2;
    timer2.start();
    for (int i = 0; i < 200; i++) {
      lb2 = lower_bound(cset1.begin(), cset1.end(), *(cset2.begin()));
    }
    timer2.stop();
    cout << "Lower bound of compressed set " << (*lb2) << endl;
    cout << "Timer: " << timer2 << endl;
    ASSERT((*lb) == (*lb2));
    cout << "Finished testIntersection..." << endl;
 }


};


#endif
