// --*- C++ -*------x---------------------------------------------------------
// $Id: Histogram4D.h,v 1.1 2006/10/18 12:10:09 bindewae Exp $
//
// Class:           Histogram
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     This class implements the concept of a histogram.
// 
// -----------------x-------------------x-------------------x-----------------

#ifndef __Histogram4D_H__
#define __Histogram4D_H__

// Includes

#include <iostream>

#include <debug.h>
#include <Vec.h>
#include <StringTools.h>
#include <HistogramStandardFunctor.h>

class Histogram4D {

public:

  /* CONSTRUCTORS */

  typedef Vec<unsigned int>::size_type size_type;
  typedef Vec<unsigned int>::size_type index_type;
  typedef Vec<Vec<Vec<Vec<double> > > > data_type;

  Histogram4D();

  Histogram4D(size_type nx1, double _minx1, double _stepx1,
	      size_type nx2, double _minx2, double _stepx2,
	      size_type nx3, double _minx3, double _stepx3,
	      size_type nx4, double _minx4, double _stepx4,
	      double pseudoCount) : minx1(_minx1), stepx1(_stepx1),
				    minx2(_minx2), stepx2(_stepx2),
				    minx3(_minx3), stepx3(_stepx3),  
				    minx4(_minx4), stepx4(_stepx4),
  x1functor(minx1, stepx1),
  x2functor(minx2, stepx2),
  x3functor(minx3, stepx3),
  x4functor(minx4, stepx4)
{   
    histData = Vec<Vec<Vec<Vec<double> > > >(nx1, 
		     Vec<Vec<Vec<double> > >(nx2,
		           Vec<Vec<double> >(nx3,
			         Vec<double>(nx4, pseudoCount) ) ) );

  }

  Histogram4D(size_type nx1, double _minx1, double _stepx1,
	      size_type nx2, double _minx2, double _stepx2,
	      size_type nx3, double _minx3, double _stepx3,
	      size_type nx4, double _minx4, double _stepx4,
	      data_type _data) : histData(_data),
				 minx1(_minx1), stepx1(_stepx1),
				 minx2(_minx2), stepx2(_stepx2),
				 minx3(_minx3), stepx3(_stepx3),  
				 minx4(_minx4), stepx4(_stepx4),
  x1functor(minx1, stepx1),
  x2functor(minx2, stepx2),
  x3functor(minx3, stepx3),
  x4functor(minx4, stepx4)
  {   
    
  }

  Histogram4D(const Histogram4D& orig);

  virtual ~Histogram4D();

  /* OPERATORS */

  /** Assigment operator. */
  Histogram4D& operator = (const Histogram4D& orig);
  /** comparison operator */
  friend bool operator==(const Histogram4D& left, const Histogram4D& right);
  /** output operator */
  friend ostream& operator << (ostream& os, const Histogram4D& object);
  /** input operator  */
  friend istream& operator >> (istream& is, Histogram4D& object);

  /* PREDICATES */

  /** return left side x-value of first bin */
  double getMinx1() const { return minx1; }
  /** return left side x-value of first bin */
  double getMinx2() const { return minx2; }
  /** return left side x-value of first bin */
  double getMinx3() const { return minx3; }
  /** return left side x-value of first bin */
  double getMinx4() const { return minx4; }

  /** return RIGHT side x-value of last bin */
  double getMaxX1() const { return minx1 + stepx1 * histData.size(); }
  /** return RIGHT side x-value of last bin */
  double getMaxX2() const { return minx2 + stepx2 * histData[0].size(); }
  /** return RIGHT side x-value of last bin */
  double getMaxX3() const { return minx3 + stepx3 * histData[0][0].size(); }
  /** return RIGHT side x-value of last bin */
  double getMaxX4() const { return minx4 + stepx4 * histData[0][0][0].size(); }


  /** return bin heights */
  data_type getData() const { return histData; }

  /** return width of bins */
  double getStepx1() const { return stepx1; }

  /** return width of bins */
  double getStepx2() const { return stepx2; }

  /** return width of bins */
  double getStepx3() const { return stepx3; }

  /** return width of bins */
  double getStepx4() const { return stepx4; }

  /** get the bin height which corresponds to the correct bin of value */
  double getValue(double x1, double x2, double x3, double x4) const;

  /** get the bin height which corresponds to the correct bin of value */
  double getSafeValue(double x1, double x2, double x3, double x4) const;


  /** get the bin height which corresponds to the correct bin of value */
  double getValue(const Vec<double>& v) const { 
    PRECOND(v.size() == 4); 
    return getValue(v[0], v[1], v[2], v[3]); 
  }

  /** get the bin height which corresponds to the correct bin of value */
  double getSafeValue(const Vec<double>& v) const { 
    PRECOND(v.size() == 4); return getSafeValue(v[0], v[1], v[2], v[3]); 
  }

//   /** Interpolate between bins. No interpolation, 
//       if value is at middle of bin */
//   double getInterpolatedValue(double orig) const;

  /** return height of bin # d */
  double getBinValue(index_type nx1, 
		     index_type nx2,
		     index_type nx3,
		     index_type nx4) const;

  /** return x-value of left side of bin */
  double getBinX1Start(unsigned int d) const { return minx1 + (stepx1 * d); }
  /** return x-value of left side of bin */
  double getBinX2Start(unsigned int d) const { return minx2 + (stepx2 * d); }
  /** return x-value of left side of bin */
  double getBinX3Start(unsigned int d) const { return minx3 + (stepx3 * d); }
  /** return x-value of left side of bin */
  double getBinX4Start(unsigned int d) const { return minx4 + (stepx4 * d); }

  size_type getDimX1() const { return histData.size(); }
  size_type getDimX2() const { return histData[0].size(); }
  size_type getDimX3() const { return histData[0][0].size(); }
  size_type getDimX4() const { return histData[0][0][0].size(); }

  /** return x-value of left side of bin */
  double getX1BinMiddle(unsigned int d) const { 
    return minx1 + (stepx1 * (d+0.5)); }
  /** return x-value of left side of bin */
  double getX2BinMiddle(unsigned int d) const { 
    return minx2 + (stepx2 * (d+0.5)); }
  /** return x-value of left side of bin */
  double getX3BinMiddle(unsigned int d) const { 
    return minx3 + (stepx3 * (d+0.5)); }
  /** return x-value of left side of bin */
  double getX4BinMiddle(unsigned int d) const { 
    return minx4 + (stepx4 * (d+0.5)); }

  bool isDefinedX1(double x) const {
    int bin = getBinX1(x);
    return ((bin >= 0) && (bin < static_cast<int>(getDimX1())));
  }
  bool isDefinedX2(double x) const {
    int bin = getBinX2(x);
    return ((bin >= 0) && (bin < static_cast<int>(getDimX2())));
  }
  bool isDefinedX3(double x) const {
    int bin = getBinX3(x);
    return ((bin >= 0) && (bin < static_cast<int>(getDimX3())));
  }
  bool isDefinedX4(double x) const {
    int bin = getBinX4(x);
    return ((bin >= 0) && (bin < static_cast<int>(getDimX4())));
  }

  bool isDefined(double x1, double x2, double x3, double x4) const {
    return isDefinedX1(x1) && isDefinedX2(x2) && isDefinedX3(x3) && isDefinedX4(x4);
  }

  bool isDefined(const Vec<double>& v) const {
    return isDefined(v[0], v[1], v[2], v[3]);
  }

  /** return bin number which corresponds to the value */
  int getBinX1(double d) const;
  /** return bin number which corresponds to the value */
  int getBinX2(double d) const;
  /** return bin number which corresponds to the value */
  int getBinX3(double d) const;
  /** return bin number which corresponds to the value */
  int getBinX4(double d) const;

  /** return bin number which corresponds to the value */
  int getSafeBinX1(double d) const;
  /** return bin number which corresponds to the value */
  int getSafeBinX2(double d) const;
  /** return bin number which corresponds to the value */
  int getSafeBinX3(double d) const;
  /** return bin number which corresponds to the value */
  int getSafeBinX4(double d) const;


  string generateInfoString() {
    string result = "" 
      + uitos(getDimX1()) + " "  
      + uitos(getDimX2()) + " "
      + uitos(getDimX3()) + " "
      + uitos(getDimX4());
    return result;
  }
  /* MODIFIERS */

  /** set overall start value */
  // void setMinx1(double x) { minx1 = x; }
  /** set overall start value */
  // void setMinx2(double x) { minx2 = x; }
  /** set overall start value */
  // void setMinx3(double x) { minx3 = x; }
  /** set overall start value */
  // void setMinx4(double x) { minx4 = x; }

  /** add data element */
  void addData(double x1, double x2, double x3, double x4) { 
    if (isDefined(x1, x2, x3, x4)) {
      (histData[getBinX1(x1)][getBinX2(x2)][getBinX3(x3)][getBinX4(x4)])++;
    }
  }

  void normalize();

  void normalizeX4();

  void normalizeX3X4();
	
  /** set data vector */
  void setData(const data_type& d) { histData = d; }

  void setFunctor1(const HistogramStandardFunctor& functor) {
    x1functor = functor;
  }
  void setFunctor2(const HistogramStandardFunctor& functor) {
    x2functor = functor;
  }
  void setFunctor3(const HistogramStandardFunctor& functor) {
    x3functor = functor;
  }
  void setFunctor4(const HistogramStandardFunctor& functor) {
    x4functor = functor;
  }

  /** set step size */
  // void setStepx1(double orig) { stepx1 = orig; }
  /** set step size */
  // void setStepx2(double orig) { stepx2 = orig; }
  /** set step size */
  // void setStepx3(double orig) { stepx3 = orig; }
  /** set step size */
  // void setStepx4(double orig) { stepx4 = orig; }

protected:
  /* OPERATORS  */
  /* PREDICATES */
  virtual bool compare(const Histogram4D& other) const;
  /* MODIFIERS  */
  virtual void copy(const Histogram4D& orig);

private:
  /* OPERATORS  */
  /* PREDICATES */
  /* MODIFIERS  */

private:
  /* PRIVATE ATTRIBUTES */
  data_type histData;
  double minx1;
  double stepx1;
  double minx2;
  double stepx2;
  double minx3;
  double stepx3;
  double minx4;
  double stepx4;
  HistogramStandardFunctor x1functor;
  HistogramStandardFunctor x2functor;
  HistogramStandardFunctor x3functor;
  HistogramStandardFunctor x4functor;
}; // class Histogram4D

#endif /* __A_CLASS_H__ */

