#include <Bed.h>
#include <debug.h>
#include <iostream>
#include <fstream>
#include <GetArg.h>
#include <vectornumerics.h>
#include <IntervalOperations.h>

typedef Bed::size_type size_type;
typedef Bed::index_type index_type;

#define DELIM "\t"

#ifndef NDEBUG
#define NDEBUG
#endif

#define INTERVOPS_VERSION "0.6.3"

/** version history
 * 0.5.1 : added intersection returning overlapping pieces of intervals (as in Galaxy)
 * 0.6.0 : logic related to filtering of pairs of intervals; reading of 3 BED files.
 * 0.6.1 : added command ijoin1i2 : inner join such that each joined first region lies completely within joined second region.
 * 0.6.2 : added way to specify annotation columns
 * 0.6.3 : added option mergename
 */

void
parseCols(string s, int& chromCol, int& startCol,int& endCol, int& nameCol, int& strandCol,
	  int& scoreCol, Vec<int>& annoCols) {
#ifndef NDEBUG
  cout << "# parseCols called with string " << s << "." << endl;
#endif
  vector<string> tokens = getTokens(s,":");
  ERROR_IF(tokens.size() < 3, "At least three columns (chrom:start:end) have to be specified: " + s);
  chromCol = stoi(tokens[0]) - 1;
  startCol = stoi(tokens[1]) - 1;
  endCol  = stoi(tokens[2]) - 1;
  if (tokens.size() >= 4) {
    nameCol = stoi(tokens[3]) - 1;
  }
  if (tokens.size() >= 5) {
    strandCol = stoi(tokens[4]) - 1;
  } 
  if (tokens.size() >= 6) {
    scoreCol = stoi(tokens[5]) - 1;
  } 
  annoCols.clear();
  if (tokens.size() >= 7) {
    for (size_type k = 6; k < tokens.size(); ++k) {
      annoCols.push_back(stoi(tokens[k-6]) - 1);
    }
  }
}

void 
versionMessage(ostream& os) {
  cout << "# intervops version " << INTERVOPS_VERSION << endl;
}

int 
main(int argc, char ** argv) {

  string filename1;
  string filename2;
  string filename3;
  string colIds1;
  string colIds2;
  string colIds3;
  int chromCol = 0;
  int chromCol2 = 0;
  int chromCol3 = 0;
  Bed::index_type indexInterval = 1000;
  int scoreCol = -1;
  int scoreCol2 = -1;
  int scoreCol3 = -1;
  int startCol = 1;
  int startCol2 = 1;
  int startCol3 = 1;
  int endCol = 2;
  int endCol2 = 2;
  int endCol3 = 2;
  int strandCol = -1;
  int strandCol2 = -1;
  int strandCol3 = -1;
  int nameCol = -1;
  int nameCol2 = -1;
  int nameCol3 = -1;
  string operation;
  int verbose = 0;
  bool writeName = true;
  bool writeStrand = true;
  bool writeScore = true;
  Vec<int> annotationCols;
  Vec<int> annotationCols2;
  Vec<int> annotationCols3;
  getArg("-c1", colIds1, argc, argv, colIds1);
  getArg("-c2", colIds2, argc, argv, colIds2);
  getArg("-c3", colIds3, argc, argv, colIds3);
  getArg("i", filename1, argc, argv, filename1);
  getArg("j", filename2, argc, argv, filename2);
  getArg("k", filename3, argc, argv, filename3);
  getArg("p", operation, argc, argv, operation);
  getArg("t", indexInterval, argc, argv, indexInterval);
  getArg("v", verbose, argc, argv, verbose);

  Bed bed1, bed2, bed3;

  if ((verbose > 0) || (argc < 2)) {
    versionMessage(cout);
  }

  if (colIds1.size() > 0) {
    parseCols(colIds1, chromCol, startCol, endCol, nameCol, strandCol, scoreCol, annotationCols);
  }
  if (colIds2.size() > 0) {
    parseCols(colIds2, chromCol2, startCol2, endCol2, nameCol2, strandCol2, scoreCol2, annotationCols2);
  }
  if (colIds3.size() > 0) {
    parseCols(colIds3, chromCol3, startCol3, endCol3, nameCol3, strandCol3, scoreCol3, annotationCols3);
  }

  ifstream file1(filename1.c_str());
  ERROR_IF(!file1, "Error opening file 1 (option -i)");
  bed1.read(file1, chromCol, startCol, endCol, strandCol, nameCol, scoreCol, annotationCols);
  file1.close();
  ERROR_IF(bed1.size() == 0, "No intervals in first interval set. Use option --c1 C1:C2:C3... to specify columns.");
  bed1.buildIndices(indexInterval); // currently not necessary
  if (verbose > 1) {
    cout << "# First region file read: " << bed1.size() << " intervals." << endl;
  }
  if (filename2.size() > 0) {
    ifstream file2(filename2.c_str());
    ERROR_IF(!file2, "Error opening file 2 (option -j)");
    bed2.read(file2, chromCol2, startCol2, endCol2, strandCol2, nameCol2, scoreCol2, annotationCols2);
    file2.close();
    if (verbose > 1) {
      cout << "# Second region file read: " << bed2.size() << " intervals." << endl;
    }
    if (indexInterval > 1) {
      if (verbose > 1) {
	cout << "# Building indices using step size " 
	     << indexInterval << " ..." << endl;
      }
      bed2.buildIndices(indexInterval);
      if (verbose > 1) {
	cout << "# Successfully built indices of second interval set." << endl;
      }
      if (verbose > 2) {
	bed2.writeIndexSets(cout);
      }
    }
  }
  if (filename3.size() > 0) {
    ifstream file3(filename3.c_str());
    ERROR_IF(!file3, "Error opening file 3 (option -j)");
    bed3.read(file3, chromCol3, startCol3, endCol3, strandCol3, nameCol3, scoreCol3, annotationCols3);
    file3.close();
    if (verbose > 1) {
      cout << "# Third region file read: " << bed3.size() << " intervals." << endl;
    }
    if (indexInterval > 1) {
      if (verbose > 1) {
	cout << "# Building indices using step size " 
	     << indexInterval << " ..." << endl;
      }
      bed3.buildIndices(indexInterval);
      if (verbose > 1) {
	cout << "# Successfully built indices of second interval set." << endl;
      }
      if (verbose > 2) {
	bed3.writeIndexSets(cout);
      }
    }
  }

  if (operation == "ijoin") {
    IntervalOperations::innerJoin(cout, bed1, bed2, verbose, writeName, writeStrand, writeScore);
  } else if (operation == "ijoin1i2") {
    IntervalOperations::innerJoin1i2(cout, bed1, bed2, verbose, writeName, writeStrand, writeScore);
  } else if (operation == "info") {
    bed1.write(cout, true, true, true);
  } else if (operation == "intersect") {
    IntervalOperations::intersect(cout, bed1, bed2, writeName, writeStrand, writeScore);
  } else if (operation == "intersectp") {
    IntervalOperations::intersectOverlappingPieces(cout, bed1, bed2, writeName, writeStrand, writeScore, verbose);
  } else if (operation == "merge") {
    IntervalOperations::merge(cout, bed1);
  } else if (operation == "mergename") {
    IntervalOperations::mergeName(cout, bed1);
  } else if (operation == "subtract") {
    IntervalOperations::subtract(cout, bed1, bed2, writeName, writeStrand, writeScore);
    // } 
  // else if (operation == "subtractboth") {
  // ERROR_IF(bed1.intervalCount() != bed2.intervalCount(),
  // "The BED format files specified by -i and -j have to have the same number of regions for operations using interval pairs.");
  // IntervalOperations::annotatePairs(cout, bed1, bed2, bed3, writeName, writeStrand, writeScore);
  } else {
    ERROR("Unknown operation use option -p (allowed: ijoin|ijoin1i2|info|intersect|intersectp|merge|mergename|subtract): " + operation);
  }
  return 0;
}
