#include <Bed.h>
#include <debug.h>
#include <iostream>
#include <fstream>
#include <GetArg.h>
#include <vectornumerics.h>
#include <IntervalOperations.h>

typedef Bed::size_type size_type;
typedef Bed::index_type index_type;

#define DELIM "\t"

#ifndef NDEBUG
#define NDEBUG
#endif

#define INTERVANN_VERSION "0.5.0"

/** version history
 * 0.5.1 : added intersection returning overlapping pieces of intervals (as in Galaxy)
 * 0.6.0 : logic related to filtering of pairs of intervals; reading of 3 BED files.
 */

void
parseCols(string s, int& chromCol, int& startCol,int& endCol, int& nameCol, int& strandCol,
	  int& scoreCol) {
  vector<string> tokens = getTokens(s,":");
  ERROR_IF(tokens.size() < 3, "At lease three columns (chrom:start:end) have to be specified: " + s);
  chromCol = stoi(tokens[0]) - 1;
  startCol = stoi(tokens[1]) - 1;
  endCol  = stoi(tokens[2]) - 1;
  if (tokens.size() >= 4) {
    nameCol = stoi(tokens[3]) - 1;
  }
  if (tokens.size() >= 5) {
    strandCol = stoi(tokens[4]) - 1;
  } 
  if (tokens.size() >= 6) {
    scoreCol = stoi(tokens[5]) - 1;
  } 
}

void 
versionMessage(ostream& os) {
  cout << "# intervann version " << INTERVANN_VERSION << endl;
}

void
helpOutput(ostream& os) {
  os << "Intervann -i INPUTFILE" << endl
     << "--c1 CHROMCOL:STARTCOL:ENDCOL    : specify columns" << endl
     << "--c2 CHROMCOL:STARTCOL:ENDCOL    : specify columns" << endl
     << "--c3 CHROMCOL:STARTCOL:ENDCOL    : specify columns" << endl
     << "-k FILENAME  : specify a third BED format file" << endl;
}

int 
main(int argc, char ** argv) {

  string filename1;
  string filename2;
  string filename3;
  string colIds1;
  string colIds2;
  string colIds3;
  int chromCol = 0;
  int chromCol2 = 0;
  int chromCol3 = 0;
  Bed::index_type indexInterval = 1000;
  int scoreCol = -1;
  int scoreCol2 = -1;
  int scoreCol3 = -1;
  int startCol = 1;
  int startCol2 = 1;
  int startCol3 = 1;
  int endCol = 2;
  int endCol2 = 2;
  int endCol3 = 2;
  int strandCol = -1;
  int strandCol2 = -1;
  int strandCol3 = -1;
  int nameCol = -1;
  int nameCol2 = -1;
  int nameCol3 = -1;
  string operation;
  int verbose = 0;
  bool writeName = true;
  bool writeStrand = true;
  bool writeScore = true;
  Vec<int> annotationCols;
  Vec<int> annotationCols2;
  Vec<int> annotationCols3; // currently not used. FIXIT
  getArg("-c1", colIds1, argc, argv, colIds1);
  getArg("-c2", colIds2, argc, argv, colIds2);
  getArg("-c3", colIds3, argc, argv, colIds3);
  getArg("i", filename1, argc, argv, filename1);
  getArg("j", filename2, argc, argv, filename2);
  getArg("k", filename3, argc, argv, filename3);
  getArg("p", operation, argc, argv, operation);
  getArg("t", indexInterval, argc, argv, indexInterval);
  getArg("v", verbose, argc, argv, verbose);

  Bed bed1, bed2, bed3;

  if ((verbose > 0) || (argc < 2)) {
    versionMessage(cout);
    helpOutput(cout);
    exit(0);
  }

  if (colIds1.size() > 0) {
    parseCols(colIds1, chromCol, startCol, endCol, nameCol, strandCol, scoreCol);
  }
  if (colIds2.size() > 0) {
    parseCols(colIds2, chromCol2, startCol2, endCol2, nameCol2, strandCol2, scoreCol2);
  }
  if (colIds3.size() > 0) {
    parseCols(colIds3, chromCol3, startCol3, endCol3, nameCol3, strandCol3, scoreCol3);
  }

  ifstream file1(filename1.c_str());
  ERROR_IF(!file1, "Error opening file 1 (option -i)");

  if (filename3.size() > 0) {
    ifstream file3(filename3.c_str());
    ERROR_IF(!file3, "Error opening file 3 (option -k)");
    bed3.read(file3, chromCol3, startCol3, endCol3, strandCol3, nameCol3, scoreCol3, annotationCols3);
    file3.close();
    if (verbose > 1) {
      cout << "# Third region file read: " << bed3.size() << " intervals." << endl;
    }
    if (indexInterval > 1) {
      if (verbose > 1) {
	cout << "# Building indices using step size " 
	     << indexInterval << " ..." << endl;
      }
      bed3.buildIndices(indexInterval);
      if (verbose > 1) {
	cout << "# Successfully built indices of second interval set." << endl;
      }
      if (verbose > 2) {
	bed3.writeIndexSets(cout);
      }
    }
  }

  if (operation == "pairs") {
     IntervalOperations::annotatePairs(file1, cout, chromCol, startCol, endCol,
				       chromCol2, startCol2, endCol2,
				       bed3, writeName, writeStrand, writeScore);
  } else {
    ERROR("Unknown operation (allowed: pairs): " + operation);
  }
  file1.close();
  return 0;
}
