// --*- C++ -*------x---------------------------------------------------------
#ifndef __SINGLE_LINKAGE_2D_PROGRESSIVE_FILTER__
#define __SINGLE_LINKAGE_2D_PROGRESSIVE_FILTER__

#include <Vec.h>
#include <algorithm>
#include <debug.h>
#include <ContainerTools.h>
#ifdef COVARNA_CONCURRENT_VECTOR
#include <tbb/concurrent_vector.h>
#endif

#ifdef COVARNA_CONCURRENT_VECTOR
 using namespace tbb;
#endif

class SingleLinkage2DProgressiveFilter  {

 public:

  typedef int length_type;
  typedef pair<length_type, length_type> pair_type;
#ifdef COVARNA_CONCURRENT_VECTOR
  typedef concurrent_vector<pair_type > result_row_type;
  typedef concurrent_vector<concurrent_vector<pair_type > > result_type;
#else
  typedef Vec<pair_type > result_row_type;
  typedef Vec<Vec<pair_type> > result_type;
#endif
  typedef Vec<length_type>::size_type size_type;

 private:

  bool active;

  bool checkDiagonalMode;

  size_type clusterCount;

  // only y-values of cluster elements are stored here. The the highest x-values of all elements of a cluster is given implicetly by its first coordinate position: 0 means it is equal to pc, 1 means it is equal to pc - 1 etc.
  Vec<Vec<Vec<length_type > > > clusterXs;

  // The corresponding x values of the cluster. 
  // the hightest x value of each cluster minus pc is equal to the first coordinate of a cluster
  Vec<Vec<Vec<length_type > > > clusterYs;

  length_type cutoff;

  length_type delay; // only activate filter, if (pc - startPc) > delay. Othewise do not require minimum size and antidiagonal 
 
  size_type discardCount;
  
  size_type elementCount; // number of stored elements

  bool flushMode;

  length_type pc; // current x-value

  bool reverseMode;
  
  size_type sizeMin; // minimum size of clusters

  length_type startPc; // very first x-value

 public:

 SingleLinkage2DProgressiveFilter(length_type _cutoff, length_type _pc) : active(true), checkDiagonalMode(true), clusterCount(0),clusterXs(_cutoff, Vec<Vec<length_type> >()),clusterYs(_cutoff, Vec<Vec<length_type> >()), cutoff(_cutoff), delay(0), discardCount(0), elementCount(0), flushMode(false), pc(_pc), reverseMode(true), sizeMin(1), startPc(_pc) {
    ASSERT(cutoff > 0);
    ASSERT(pc >= 0);
 }

 /** Copy constructor */
 SingleLinkage2DProgressiveFilter(const SingleLinkage2DProgressiveFilter& other) { copy(other); }

 virtual ~SingleLinkage2DProgressiveFilter() { }

  virtual SingleLinkage2DProgressiveFilter& operator = (const SingleLinkage2DProgressiveFilter other) {
    if (&other != this) {
      copy(other);
    }
    return *this;
  }

  virtual void copy(const SingleLinkage2DProgressiveFilter& other) {
    active = other.active;
    checkDiagonalMode = other.checkDiagonalMode;
    clusterCount = other.clusterCount;
    clusterXs = other.clusterXs;
    clusterYs = other.clusterYs;
    cutoff = other.cutoff;
    delay = other.delay;
    discardCount = other.discardCount; 
    elementCount = other.elementCount; // number of stored elements
    flushMode = other.flushMode;
    pc = other.pc; // current x-value
    reverseMode = other.reverseMode;
    sizeMin = other.sizeMin; // minimum size of clusters
    startPc = other.startPc;
  }

  virtual result_type flushAll() {
    // DEBUG_MSG("Called method flush");
    flushMode = true;
    result_type result = movePc(pc + cutoff + 1);
    pc -= (cutoff + 1);
    flushMode = false;
    POSTCOND(getClusterCount() == 0);
    POSTCOND(getElementCount() == 0);
    // DEBUG_MSG("Finished method flush");
    return result;
  }

  /** Same as flush, but returns only clusters that have minimum size */
  virtual result_type flushBest() {
    // DEBUG_MSG("Called method flush");
    result_type result = movePc(pc + cutoff + 1);
    pc -= (cutoff + 1);
    POSTCOND(getClusterCount() == 0);
    POSTCOND(getElementCount() == 0);
    // DEBUG_MSG("Finished method flush");
    return result;
  }

  virtual length_type getCutoff() { return cutoff; }

  virtual size_type getClusterCount() const { return clusterCount; }

  virtual length_type getDelay() const { return delay; }

  virtual size_type getDiscardCount() const { return discardCount; }

  virtual size_type getElementCount() const { return elementCount; }

  virtual bool isActive() const { return active; }

  virtual bool isCheckDiagonalMode() const { return checkDiagonalMode; }

  /** Resets - it deleted information from previous search. The active status, however, is unchanged. */
  virtual void reset() {
    flushAll();
    discardCount = 0;
    pc = 0;
    POSTCOND(getElementCount() == 0 && getClusterCount() == 0 && getDiscardCount() == 0);
  }

  virtual void setReverseMode(bool mode) { reverseMode = mode; }

  virtual size_type size() const { return clusterYs.size(); }

  /** Central method that adds an element. One condition is that the x values are never decreasing, unless "reset" is being called. 
   * Based on the new x-value, existing clusters are being
   * "finalized". If they have the minimum size they are being returned,
   * otherwise they are being deleted.
   * Clusters are shifted appropriately if a new x-value is given.
   * An appropriate cluster is found for the x-y pair.
   * If no appropriate cluster is found, a new cluster is being started. */
  virtual result_type push(length_type x, length_type y)  {
    // cout << "Called method push with : " << x <<  " " << y << endl;
    result_type result;
    if (!active) { // simply pass through
      result.push_back(result_row_type(1, pair<length_type, length_type>(x, y))); 
      ASSERT(result.size() == 1);
      ASSERT(result[0].size() == 1);
      return result;
    }
    if (x != pc) { // new x-position
      result =  movePc(x);
      ASSERT(pc == x);
    }
    add(y);

    return result;
  }

  virtual void setActive(bool _active) { active = _active; }

  virtual void setCheckDiagonalMode(bool mode) {
    checkDiagonalMode = mode;
  }

  virtual void setDelay(length_type _delay) { delay = _delay; }

  virtual void setSizeMin(size_type _sizeMin) { sizeMin = _sizeMin; }

  virtual bool validate() const {
    return (clusterXs.size() == clusterYs.size()) && (clusterXs.size() == static_cast<size_type>(cutoff)) && (cutoff > 0)
      && (elementCount >= 0);
  }

  virtual void write(ostream& os) const {
    os << size() << " " << clusterCount << " " << elementCount << endl;
    for (size_type i = 0; i < size();++i) {
      if (clusterYs[i].size() > 0) {
	os << i << " : ";
	for (size_type j = 0; j < clusterYs[i].size(); ++j) {
	  for (size_type k = 0; k < clusterYs[i][j].size(); ++k) {
	    os << " " << clusterXs[i][j][k] << "," << clusterYs[i][j][k];
	  }
	  os << " ; ";
	}
	os << endl;
      }
    }
  }

  static void writeClusters(const result_type& clusters) {
    for (size_type i = 0; i < clusters.size(); ++i) {
      cout << "Cluster "<< (i+1) << " : " << endl;
      for (size_type j = 0; j < clusters[i].size(); ++j) {
	cout << clusters[i][j].first << " " << clusters[i][j].second << endl;
      }
    }
  }


 private:

  /** The x-value is assumed to be equal to pc */
  virtual void add(length_type y) {
    // DEBUG_MSG("Started method add...");
    // cout << "Add: Adding " << y << " with pc " << pc << endl;
    ASSERT(validate());
    ++elementCount;
    for (size_type i = 0; i < clusterYs.size(); ++i) {
      for (size_type j = 0; j < clusterYs[i].size(); ++j) {
	ASSERT(clusterYs[i][j].size()>0);
	ASSERT(clusterYs[i][j].size() == clusterXs[i][j].size());
	if ((y >= (clusterYs[i][j][0] - cutoff))
	    && (y <= (clusterYs[i][j][clusterYs[i][j].size()-1] + cutoff))) {
	  // correct cluster found! No insert it such that it keeps being sorted
	  // cout << "Appropriate cluster found! So far: " << clusterYs[i][j] << endl << clusterXs[i][j] << endl;
	  Vec<length_type>::iterator lb = lower_bound(clusterYs[i][j].begin(), clusterYs[i][j].end(),y);
	  if (lb == clusterYs[i][j].end()) {
	    clusterXs[i][j].insert(clusterXs[i][j].end(), pc);
	  } else {
	    clusterXs[i][j].insert(clusterXs[i][j].begin() + distance(clusterYs[i][j].begin(),lb), pc);
	  }
	  // clusterXs[i][j].insert(clusterXs[i][j].begin() + 1, pc);
	  // cout << "Inserting before position " << distance(clusterYs[i][j].begin(),lb) << endl;
          ASSERT(ContainerTools::isSorted(clusterYs[i][j].begin(), clusterYs[i][j].end()));
	  clusterYs[i][j].insert(lb, y);
          if (!ContainerTools::isSorted(clusterYs[i][j].begin(), clusterYs[i][j].end())) {
	    cout << "Found unsorted y array: " << i << " " << j <<  " : " << clusterYs[i][j] << endl;
	  }
          ASSERT(ContainerTools::isSorted(clusterYs[i][j].begin(), clusterYs[i][j].end()));
	  // DEBUG_MSG("So far added x and y values");
	  // cout << clusterXs[i][j] << clusterYs[i][j] << endl;
	  // x-positions to not have to be sorted
          // ASSERT(ContainerTools::isSorted(clusterXs[i][j].begin(), clusterXs[i][j].end()));
	  ASSERT(validate());
	  clusterXs[0].push_back(clusterXs[i][j]);
	  clusterYs[0].push_back(clusterYs[i][j]);
	  // DEBUG_MSG("erasing...");
	  //	  clusterXs[i][j].erase(clusterXs[i][j].begin(),clusterXs[i][j].end());
	  //  clusterYs[i][j].erase(clusterYs[i][j].begin(),clusterYs[i][j].end());
	  clusterXs[i].erase(clusterXs[i].begin() + j);
	  clusterYs[i].erase(clusterYs[i].begin() + j);
	  ASSERT(validate());
	  // DEBUG_MSG("Finished method add (2)...");
	  return;
	}
      }
    }
    // no cluster found, start new cluster:
    clusterYs[0].push_back(Vec<length_type>(1, y));
    clusterXs[0].push_back(Vec<length_type>(1, pc));
    ++clusterCount;
    ASSERT(validate());
    // DEBUG_MSG("Finished method add...");
  }

  /** If a new element with a new x value is added,
   * it potentially leads to a shift of all clusters.
   * Too distance clusters are finalized (deleted) and returned as result.
   */
  virtual result_type movePc(length_type x) {
    // DEBUG_MSG("Started method movePc");
    ASSERT(x >= pc);
    result_type result;
    length_type diff = x - pc;
    if (diff == 0) {
      return result;
    }
    size_type iStart = 0;
    if (static_cast<length_type>(clusterYs.size()) > diff) {
      iStart = clusterYs.size()-diff;
    }
    for (size_type i = iStart; i < clusterYs.size(); ++i) {
      for (size_type j = 0; j < clusterYs[i].size(); ++j) {
        ASSERT(clusterYs[i][j].size() > 0);
        ASSERT(pc >= startPc);
        ASSERT((delay > 0) || ((pc - startPc) >= delay)); // delay is only active if greater zero 
	if ((clusterYs[i][j].size() >= sizeMin) && ((!checkDiagonalMode) || checkPairsHaveDiagonal(clusterXs[i][j], clusterYs[i][j]))) {
	  result.push_back(makePairs(clusterXs[i][j], clusterYs[i][j]));
	} else if (flushMode || ((delay > 0) && ((pc - startPc) < delay))) {
	  result.push_back(makePairs(clusterXs[i][j], clusterYs[i][j]));
	} else {
	  discardCount += clusterYs[i][j].size();
	}
	elementCount -= clusterXs[i][j].size();
	ASSERT(clusterCount > 0);
	--clusterCount;
	ASSERT(elementCount >= 0);
      }
      clusterXs[i].clear();
      clusterYs[i].clear();
    }
    if (diff < cutoff) {
      rotate(clusterXs.begin(), (clusterXs.begin()+diff), clusterXs.end());
      rotate(clusterYs.begin(), (clusterYs.begin() + diff), clusterYs.end());
    }
    pc = x;
    POSTCOND(pc == x);
    // DEBUG_MSG("Finished method movePc");
    return result;
  }
  
  static result_row_type makePairs(const Vec<length_type>& xv,
				   const Vec<length_type>& yv) {
    ASSERT(xv.size() == yv.size());
    result_row_type result(xv.size());
    for (size_type i = 0; i < xv.size(); ++i) {
      result[i] = pair<length_type, length_type>(xv[i], yv[i]);
    }
    return result; 
  }

  /** See if at least one anti-diagonal is found */
  bool checkPairsHaveDiagonal(const Vec<length_type>& xv,
			      const Vec<length_type>& yv) {
    ASSERT(xv.size() == yv.size());
    set<length_type> invariants;
    for (size_type i = 0; i < xv.size(); ++i) {
      if (!reverseMode) { // instead of reverse complements, match forward-matches 
	invariants.insert(yv[i] - xv[i]); // difference is invariant of base pairs in "forward-match" stems 
      } else { // reverse complements: sum of indices is invariant
	invariants.insert(xv[i] + yv[i]); // sum is invariant of base pairs in stem 
      }
    }
    return (invariants.size() < xv.size()); // At least one invariant must be duplicate. This leads to a set size that is smaller than the input size.
  }


};

#endif
