#include <iostream>
#include <fstream>
#include <debug.h>
#include <Vec.h>
#include <GetArg.h>

using namespace std;

typedef int int_type;

/** Find score that corresonds to a position. 
 * @param starts: start of intervals (0-based)
 * @param ends : ends of intervals (1-based)
 * @param scores: score valid for each position of an interval
 * @FIXIT increase speed by using lower_bound etc algorithms.
 */
double
findScore(const Vec<int_type>& starts,
	  const Vec<int_type>& ends,
	  const Vec<double>& scores,
	  int_type position,
	  double defaultScore) {
  ASSERT(starts.size() > 0);
  ASSERT(starts.size() == ends.size());
  ASSERT(starts.size() == scores.size());
  Vec<int_type>::size_type n = starts.size();
  if ((position >= ends[n - 1]) || (position < starts[0])) {
    return defaultScore;
  }
  for (Vec<int_type>::size_type i = 0; i < n; ++i) {
    if ((position >= starts[i]) && (position < ends[i])) {
      return scores[i];
    }
  }
  return defaultScore;
}

/** Central function that performs binning */
Vec<Vec<double> > 
compileScores(const Vec<int_type>& starts,
	      const Vec<int_type>& ends,
	      const Vec<double>& scores,
	      const Vec<int_type>& queries,
	      int_type range,
	      double defaultScore) {
  ASSERT(starts.size() == ends.size());
  Vec<Vec<double> > result;
  // offset
  for (int_type offs = -range; offs <= range; offs++) {
    Vec<double> foundScores;
    foundScores.clear(); 
    for (Vec<int_type>::size_type i = 0; i < queries.size(); ++i) {
      double score = findScore(starts, ends, scores, queries[i] + offs,
			       defaultScore);
      foundScores.push_back(score);
    }
    result.push_back(foundScores);
  }
  POSTCOND(static_cast<int_type>(result.size()) == (2*range + 1));
  return result;
}

void 
helpOutput(ostream& os) {
  os << "Program regionscores compiles interval scores for a range of query positions." << endl;
  os << "Option: regionscores -i regionfile -j queryfile -c chromosome -r range -v verbose" << endl;
  os << "Example:" << endl;
  os << "./regionscores -c chr1 -i ../../test/fixtures/regionscores_test1_a.dat -j ../../test/fixtures/regionscores_test1_b.dat -v 3" << endl;
}

/** Main program */
int main(int argc, char ** argv) {
  double defaultScore = 0;
  string filename, filename2;
  Vec<int_type> starts;
  Vec<int_type> ends;
  Vec<double> scores;
  Vec<int_type> queries;
  int_type range = 20;
  int verbose = 0;
  string chromName;
  getArg("i", filename, argc, argv, filename);
  getArg("j", filename2, argc, argv, filename2);
  getArg("c", chromName, argc, argv,chromName);
  getArg("r", range, argc, argv, range);
  getArg("v", verbose, argc, argv, verbose);
  int_type val1, val2;
  double val3;
  string name;
  if (filename.size() == 0 || filename2.size() == 0) {
    helpOutput(cout);
    exit(0);
  }
  if (verbose > 0) {
    cerr << "# Starting with " << chromName << endl;
  }
  ifstream inputFile(filename.c_str());
  ERROR_IF(!inputFile, "Error opening input file (option -i).");
  while (inputFile) {
    inputFile >> name >> val1 >> val2 >> val3;
    if (inputFile) {
      if ((chromName.size() == 0) || (name == chromName)) {
	starts.push_back(val1);
	ends.push_back(val2);
	scores.push_back(val3);
      }
    }
  }
  inputFile.close();
  ifstream inputFile2(filename2.c_str());
  ERROR_IF(!inputFile2, "Error opening input file 2 (option -j).");
  while (inputFile2) {
    inputFile2 >> name >> val1;
    if (inputFile2) {
      if ((chromName.size() == 0) || (name == chromName)) {
	queries.push_back(val1);
      }
    }
  }
  inputFile2.close();
  if (verbose > 1) {
    cout << "# Start positions: " << starts;
    cout << "# End positions: " << ends;
  }
  Vec<Vec<double> > result = compileScores(starts, ends, scores, 
			         queries, range, defaultScore);
  
  for (Vec<Vec<double> >::size_type i = 0; i < result.size(); ++i) {
    cout << (static_cast<int_type>(i) - range);
    for (Vec<double>::size_type j = 0; j < result[i].size(); ++j) {
      cout << " " << result[i][j];
    }
    cout << endl;
  }
  if (verbose > 0) {
    cout << "# Good bye!" << endl;
  }
  return 0;
}


