// --*- C++ -*------x---------------------------------------------------------
// $Id: matrixaverager.cc,v 1.6 2008/12/13 16:03:13 bindewae Exp $
//
// Program:         - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2008/12/13 16:03:13 $
//
// Description:     
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <fstream>
#include <string>
#include <Vec.h>
#include <debug.h>
#include <GetArg.h>
#include <FileName.h>
#include <vectornumerics.h>

void
helpOutput(ostream& os)
{
  os << "usage: addmatrices -i inputfiles -m [1|2] --verbose n" << endl;
  os << "--average 0|1|2 : no averaging , average, standard deviation" << endl;
  os << "--if inputformat" << endl
     << "--of outputformat" << endl
     << "--std mul" << endl
     << endl;
  
}

/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

/** reads matrix in format that is also used by program "R" */
/*
Vec<Vec<double> >
readPlainMatrix(istream& is)
{
  Vec<Vec<double > > result;
  while (is) {
    string line = getLine(is);
    Vec<string> words = getTokens(line);
    if (words.size() > 0) {
      Vec<double> row(words.size());
      for (unsigned int i = 0; i < words.size(); ++i) {
	row[i] = stod(words[i]);
	if (!isDefined(row[i])){
	  row[i] = 0.0;
	}
      }
      result.push_back(row);
    }
  }
  return result;
}
*/

void
writeMatrix(ostream& os,
	    const  Vec<Vec<double> >& m)
{
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m.size(); ++j) {
      os << m[i][j] << " ";
    }
    os << endl;
  }
}

double
elementDivision(double enu, double denom) {
  if (denom == 0.0) {
    return 0.0;
  }
  return enu / denom;
}

/** returns log of fraction or penalty values in case of undefined values */
double
elementDivisionLog(double enu, double denom,
		   double enumeratorPenalty,
		   double denominatoryPenalty,
		   double bothPenalty) {
  if (enu <= 0.0) {
    if (denom <= 0.0) {
      return bothPenalty;
    }
    return enumeratorPenalty;
  }
  else if (denom <= 0.0) {
    return denominatoryPenalty;
  }
  double frac = elementDivision(enu, denom);
  ASSERT(frac > 0.0);
  return log(frac);
}

/** returns mean of element posX, posY average over matrices */
double
elementMean(const Vec<Vec<Vec<double> > >& matrices,
	    Vec<Vec<double> >::size_type posX,
	    Vec<double>::size_type posY) {
  PRECOND(matrices.size() > 0);
  double sum = 0.0;
  for (Vec<Vec<Vec<double> > >::size_type k = 0; k < matrices.size(); ++k) {
    sum += matrices[k][posX][posY];
  }
  return sum / matrices.size();
}

/** returns variance of element posX, posY matrices */
double
elementVariance(const Vec<Vec<Vec<double> > >& matrices,
	    Vec<Vec<double> >::size_type posX,
	    Vec<double>::size_type posY) {
  PRECOND(matrices.size() > 0);
  double sum = 0.0;
  double sumSquares = 0.0;
  double term;
  for (Vec<Vec<Vec<double> > >::size_type k = 0; k < matrices.size(); ++k) {
    term = matrices[k][posX][posY];
    sum += term;
    sumSquares += term * term;
  }
  return varianceFromSum(sum, sumSquares, matrices.size());
}

/** returns mean of element posX, posY average over matrices,
 * zero if mean is not stdMul standard deviations different from zero. */
double
elementSignificantMean(const Vec<Vec<Vec<double> > >& matrices,
		       Vec<Vec<double> >::size_type posX,
		       Vec<double>::size_type posY,
		       double stdMul) {
  PRECOND(matrices.size() > 0);
  double result = elementMean(matrices, posX, posY);
  double std = sqrt(elementVariance(matrices, posX, posY));
  if (fabs(result) < (stdMul * std)) {
    return 0.0; // result not considered significant
  }
  return result;
}


/** Prints various output formats in element wise fashion */
void
elementOutput(ostream& os,
	      const Vec<Vec<Vec<double> > >& matrices,
	      int outputFormat,
	      double stdMul) {
  Vec<Vec<double> >::size_type dimx = matrices[0].size();
  Vec<double>::size_type dimy = matrices[0][0].size();
  // double val;
  for (Vec<Vec<double> >::size_type i = 0; i < dimx; ++i) {
    for (Vec<double>::size_type j = 0; j < dimy; ++j) {
      switch (outputFormat) {
      case 4:
	os << (matrices[0][i][j] - matrices[1][i][j]) << " ";
	break;
      case 5:
	os << elementDivision(matrices[0][i][j], matrices[1][i][j]) << " ";
	break;
      case 6:
	os << elementDivisionLog(matrices[0][i][j], matrices[1][i][j], 0.0, 0.0, 0.0) << " ";
	break;
      case 7:
	os << elementMean(matrices, i, j) << " "; // actual average matrix
	break;
      case 8:
	os << elementSignificantMean(matrices, i, j, stdMul) << " "; // actual average matrix, zero if not stdMul standard deviations from zero
	break;
      default:
	ERROR("Unknown matrix element output format!");
      }
    }
    os << endl;
  }
}

int
main(int argc, char ** argv)
{

  bool helpMode;
  int argcFile = 0;
  int mode = 1;
  char ** argvFile = 0;
  int avgMode = 0;
  int inputFormat = 3;
  int outputFormat = 2;
  double stdMul = 2.0; // how many standard deviations are considered statistically significant?
  unsigned int verboseLevel = 1;
  string commandFileName;
  string logFileName; //  = "mainprogramtemplate.log";
  string rootDir = ".";
  Vec<string> inputFileNames;

  getArg("-help", helpMode, argc, argv);

  if ((argc < 2) || helpMode)  {
    helpOutput(cout);
    exit(0);
  }

  getArg("-root", rootDir, argc, argv, rootDir);
  addSlash(rootDir);

  getArg("-commands", commandFileName, argc, argv, commandFileName);
  addPathIfRelative(commandFileName, rootDir);

  if (commandFileName.size() > 0) {
    ifstream commandFile(commandFileName.c_str());
    if (!commandFile) {
      if (isPresent("-commands", argc, argv)) {
	ERROR_IF(!commandFile, "Error opening command file.");
      }
      else {
	cerr << "Warning: Could not find command file: " + commandFileName 
	     << endl;
      }
    }
    else {
      argvFile = streamToCommands(commandFile, argcFile, 
				  string("mainprogramtemplate"));
    }
    commandFile.close();
  }
  
  getArg("-average", avgMode, argc, argv, avgMode);
  getArg("i", inputFileNames, argc, argv);
  getArg("-if", inputFormat, argc, argv, inputFormat);
  getArg("-log", logFileName, argc, argv, logFileName);
  getArg("-log", logFileName, argcFile, argvFile, logFileName);
  addPathIfRelative(logFileName, rootDir);
  getArg("m", mode, argcFile, argvFile, mode);
  getArg("m", mode, argc, argv, mode);
  getArg("-std", stdMul, argc, argv, stdMul);
  getArg("-of", outputFormat, argc, argv, outputFormat);
  getArg("-verbose", verboseLevel, argcFile, argvFile, verboseLevel);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);


  if (logFileName.size() > 0) {
    ofstream logFile(logFileName.c_str(), ios::app);
    parameterOutput(logFile, argc, argv);
    if (argcFile > 1) {
      logFile << "Parameters from command file: ";
      parameterOutput(logFile, argcFile, argvFile);
    }
    logFile.close();
  }


  /***************** MAIN PROGRAM *****************************/

  Vec<Vec<Vec<double> > > matrices(inputFileNames.size());
  Vec<Vec<double > > tmpMatrix;  
  for (unsigned int i = 0; i < inputFileNames.size(); ++i) {
    if (verboseLevel > 1) {
      cout << "Reading matrix : " << inputFileNames[i] << endl;
    }
    ifstream inputFile(inputFileNames[i].c_str());
    ERROR_IF(!inputFile, "Error opening input file!");
    switch (inputFormat) {
    case 1: inputFile >> matrices[i];
      break;
    case 3: matrices[i] = readPlainMatrix(inputFile);
      break;
    default: ERROR("Unknown input format!");
    }
    inputFile.close();
    ERROR_IF(matrices[i].size() != matrices[0].size(),
	     "Matrices must have same dimensions!");
  }

  Vec<Vec<double> >::size_type dimx = matrices[0].size();
  Vec<double>::size_type dimy = matrices[0][0].size();

  /*
  if (avgMode == 1) { // compute average
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	matrix[i][j] /= inputFileNames.size();
      }
    }
  }
  else if (avgMode == 2) { // compute standard deviation
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	matrix[i][j] = sqrt(varianceFromSum(matrix[i][j],
			       matrixSquare[i][j],inputFileNames.size()));
      }
    }
  }
  else if (avgMode == 3) { // average unless zero hypothesis cannot be rouled out:
    for (unsigned int i = 0; i < matrix.size(); ++i) {
      for (unsigned int j = 0; j < matrix[i].size(); ++j) {
	double dev = sqrt(varianceFromSum(matrix[i][j],
			       matrixSquare[i][j],inputFileNames.size()));
	double avg = matrix[i][j] / inputFileNames.size();
	if (fabs(avg) <= dev) {
	  matrix[i][j] = 0.0;
	}
	else {
	  matrix[i][j] /= inputFileNames.size();
	}
      }
    }

  }
  else {
    ERROR("Unknown averaging mode!");
  }
  */
  // double tmp = 0.0;
  switch (outputFormat) {
  case 1: cout << matrices << endl;
    break;
  case 2:
    for (unsigned int i = 0; i < dimx; ++i) {
      for (unsigned int j = 0; j < dimy; ++j) {
	cout << matrices.size() << " " << i << " " << j << " ";
	for (unsigned int k = 0; k < matrices.size(); ++k) {
	  cout << matrices[k][i][j] << " ";
	}
	cout << endl;
      }
    }
    break;
  case 3:  // output of triangle
    for (unsigned int i = 0; i < dimx; ++i) {
      for (unsigned int j = i; j < dimy; ++j) {
	cout << matrices.size() << " " << i << " " << j << " ";
	for (unsigned int k = 0; k < matrices.size(); ++k) {
	  cout << matrices[k][i][j] << " ";
	}
	cout << endl;
      }
    }
    break;
  default:
    elementOutput(cout, matrices, outputFormat, stdMul);
  }

  return 0;
}
