#ifndef __CLUSTER_ALGORITHMS__
#define __CLUSTER_ALGORITHMS__

#include <Vec.h>
#include <Vector3D.h>
#include <vectornumerics.h>

/** returns true if two datavectors are more similar (abs norm) than cutoff */
bool
isDataVecSimilar(const Vec<double>& dataVec,
		 const Vec<double>& lastVec, 
		 const Vec<unsigned int>& mask, 
		 double cutoff);

Vec<unsigned int>
inline
clusterRepresentatives(const Vec<Vec<unsigned int> >& clusters)
{
  Vec<unsigned int> result(clusters.size());
  for (unsigned int i = 0; i < result.size(); ++i) {
    result[i] = clusters[i][0];
  }
  return result;
}

Vec<Vec<unsigned int> >
inline
clusterRandomSubsets(const Vec<Vec<unsigned int> >& clusters,
		     unsigned int maxSize)
{
  Vec<Vec<unsigned int> > result(clusters.size());
  unsigned int useSize = maxSize;
  for (unsigned int i = 0; i < result.size(); ++i) {
    useSize = maxSize;
    if (clusters[i].size() < useSize) {
      useSize = clusters[i].size();
    }
    Vec<unsigned int> subs = generateRandomIndexSubset(useSize, 
						       clusters[i].size(), 0);
    result[i] = getSubset(clusters[i], subs);
  }
  return result;
}


/** return clustered entities , given a distance matrix and a cutoff */
Vec<Vec<unsigned int> > 
completeLinkage(const Vec<Vec<double> >& field,	double cutoff);

/** return clustered entities , given a distance matrix and a cutoff */
Vec<Vec<unsigned int> > 
singleLinkage(const Vec<Vec<double> >& field,	double cutoff);

/** return clustered entities , given a vector of 1d values and a cutoff. Fast: no distance matrix is generated. */
Vec<Vec<unsigned int> > 
singleLinkage1d(const Vec<double>& values, double cutoff);

/** return clustered entities , given a vector of 1d values and a cutoff. Fast: no distance matrix is generated. */
bool
singleLinkage1dTest();

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff.
    Special version, which avoids generation of matrix.
*/
Vec<Vec<unsigned int> > singleLinkage(const Vec<Vector3D>& v,	double cutoff);

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff to a (quite arbitrary) representative */
Vec<Vec<unsigned int> > simpleRepresentativeLinkage(const Vec<Vector3D>& v, 
						    double cutoff);

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > 
simpleRepresentativeLinkage(const Vec<Vec<double> >& v, double cutoff);

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > 
simpleSortLinkage(const Vec<Vec<double> >& v, double cutoff);

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  unsigned int k);

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  unsigned int k,
		  const Vec<double>& scaling);

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  unsigned int k, 
		  const Vec<double>& scaling,
		  const Vec<Vec<unsigned int> >& clusters,
		  double clusterCutoff);

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  const Vec<Vec<Vec<double> > >& clustData,
		  unsigned int k, 
		  const Vec<double>& scaling,
		  const Vec<Vec<unsigned int> >& clusters,
		  const Vec<Vec<Vec<unsigned int> > >& subClusters,
		  double clusterCutoff,
		  double clusterCutoff2);

Vec<unsigned int>
knnThin(Vec<Vec<double> > data,
	Vec<int> classes,
	unsigned int k);

Vec<unsigned int>
knnThin(Vec<Vec<double> > data,
	Vec<unsigned int> classes,
	unsigned int k);

Vec<unsigned int>
uniquify(const Vec<Vec<double> >& data,
	 Vec<unsigned int> mask,
	 unsigned int simMax,
	 double cutoff);

#endif
