#include <clusterAlgorithms.h>
#include <vectornumerics.h>
#include <PurgePriorityQueue.h>
#include <RankedSolution3.h>
#include <Random.h>

// #define DEBUG_VERBOSE

// Cluster algorithms:

/** largest distance to any vector of cluster is distance to point v */
double
distToClusterCompleteLinkage(unsigned int v,
	    const Vec<unsigned int>& cluster,
	    const Vec<Vec<double> >& field)
{
  double result = 0.0;
  
  for (unsigned int i = 0; i < cluster.size(); ++i) {
    if (field[cluster[i]][v] > result) {
      result = field[cluster[i]][v];
    }
  }
  return result;
}

/** smalles distance to any vector of cluster is distance to point v */
double
distToClusterSingleLinkage(unsigned int v,
	    const Vec<unsigned int>& cluster,
	    const Vec<Vec<double> >& field)
{
  PRECOND(cluster.size() > 0);
  double result = field[cluster[0]][v];
  for (unsigned int i = 1; i < cluster.size(); ++i) {
    if (field[cluster[i]][v] < result) {
      result = field[cluster[i]][v];
    }
  }
  return result;
}

unsigned int
bestClusterCompleteLinkage(unsigned int v,
	    const Vec<Vec<unsigned int> > clusters,
	    const Vec<Vec<double> >& field, double cutoff)
{
  double bestDist = cutoff;
  unsigned int bestId = clusters.size();
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    double d = distToClusterCompleteLinkage(v, clusters[i], field);
    if (d < bestDist) {
      bestDist = d;
      bestId = i;
    }
  }
  return bestId;
}

unsigned int
bestClusterSingleLinkage(unsigned int v,
	    const Vec<Vec<unsigned int> > clusters,
	    const Vec<Vec<double> >& field, double cutoff)
{
  double bestDist = cutoff;
  unsigned int bestId = clusters.size();
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    double d = distToClusterSingleLinkage(v, clusters[i], field);
    if (d < bestDist) {
      bestDist = d;
      bestId = i;
    }
  }
  return bestId;
}

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > completeLinkage(
	const Vec<Vec<double> >& field,	double cutoff)
{
  
  Vec<Vec<unsigned int> > clusters;
  Vec<unsigned int> zeroVec;
  for (unsigned int i = 0; i < field.size(); ++i) {
    unsigned int best = bestClusterCompleteLinkage(i,clusters,field, cutoff);
    if (best >= clusters.size()) {
      clusters.push_back(zeroVec);
      clusters[clusters.size()-1].push_back(i);
    }
    else {
      clusters[best].push_back(i);
    }
  }
  return clusters;
}

/** computees average distance of entry in cluster */
double
avgDistCluster(Vec<unsigned int>& cluster,
	    unsigned int id,
	    const Vec<Vec<double> >& field) {
  double avgDist = 0;
  for (Vec<unsigned int>::size_type i = 0; i < cluster.size(); ++i) {
    double term = field[cluster[i]][cluster[id]];
    avgDist += term * term;
  }
  avgDist /= cluster.size();
  avgDist = sqrt(avgDist); // use square approach
  return avgDist;
}

/** sorts clusters such that first entry is representative */
unsigned int
findBestClusterMember(Vec<unsigned int>& cluster,
		      const Vec<Vec<double> >& field) {
  double bestAvg = 1e30;
  unsigned int bestId = 0;
  for (unsigned int i = 0; i < cluster.size(); ++i) {
    double avg = avgDistCluster(cluster, i, field);
    if ((i == 0) || (avg < bestAvg)) {
      bestAvg = avg;
      bestId = i;
    }
  }
  return bestId;
}

/** sorts clusters such that first entry is representative */
void
sortCluster(Vec<unsigned int>& cluster,
	    const Vec<Vec<double> >& field) {
  unsigned int id = findBestClusterMember(cluster, field);
  unsigned int tmp = cluster[0];
  cluster[0] = cluster[id];
  cluster[id] = tmp; // swap content
}

/** sorts clusters such that first entry is representative */
void
sortClusters(Vec<Vec<unsigned int> >& clusters,
	     const Vec<Vec<double> >& field) {
  for (unsigned int i =0; i < clusters.size(); ++i) {
    sortCluster(clusters[i], field);
  }
}

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > singleLinkage(
	const Vec<Vec<double> >& field,	double cutoff)
{
  
  Vec<Vec<unsigned int> > clusters;
  Vec<unsigned int> zeroVec;
  for (unsigned int i = 0; i < field.size(); ++i) {
    unsigned int best = bestClusterSingleLinkage(i,clusters,field, cutoff);
    if (best >= clusters.size()) {
      // no appropriate cluster found, start new cluster
      clusters.push_back(zeroVec);
      clusters[clusters.size()-1].push_back(i);
    }
    else {
      // add to closest cluster
      clusters[best].push_back(i);
    }
  }
  sortClusters(clusters, field); // best representative is first!
  return clusters;
}

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > singleLinkage1d(const Vec<double>& values, double cutoff)
{
  ASSERT(values.size() > 0);
  Vec<Vec<unsigned int> > clusters;
  Vec<unsigned int> order = documentedOrder(values); 
  // cout << "Order of values " << values << " : " << order << endl;
  clusters.push_back(Vec<unsigned int>(1,order[0])); // start first cluster
  Vec<unsigned int> zeroVec;
  for (unsigned int i = 1; i < order.size(); ++i) {
    double oldVal = values[order[i-1]];
    double currVal = values[order[i]];
    ASSERT(currVal >= oldVal); // should be sorted
    double dist = currVal - oldVal;
    if (dist > cutoff) {
      // cout << "Starting new cluster because value " << currVal << " is great old value: " << oldVal << endl;
      clusters.push_back(zeroVec); // start new cluster
    }
    clusters[clusters.size()-1].push_back(order[i]); // add to most current cluster
  }
  return clusters;
}

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
bool singleLinkage1dTest() 
{
  Vec<double> values1;
  Random& rnd = Random::getInstance(); 
  values1.push_back(2.0);
  values1.push_back(3.0);
  values1.push_back(7.0);
  values1.push_back(28.0);
  values1.push_back(30.0);
  random_shuffle(values1.begin(), values1.end(), rnd);
  double cutoff = 10.0;
  Vec<Vec<unsigned int> > clusters = singleLinkage1d(values1, 10.0);
  cout << "Testing singleLinkage1d function with cutoff " << cutoff << " and input vector " <<  values1 << " Result: " << endl;
  cout << clusters << endl;
  return ((clusters.size() == 2) && (clusters[0].size() == 3) && (clusters[1].size() == 2));
}


/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > singleLinkage(const Vec<Vector3D>& v,	double cutoff)
{
  const int UNCLASSIFIED = -1;
  double cutoffSqr = cutoff * cutoff;
  Vec<Vec<unsigned int> > clusters;
  Vec<unsigned int> zeroVec;
  // cluster.push_back(zeroVec);
  // clusters[0].push_back(v[0]);
  Vec<int> takenCareOf(v.size(), UNCLASSIFIED);
  // takenCareOf[0] = 0; // point zero is put into cluster zero
  unsigned int pointsClassified = 0; 
  while (pointsClassified < v.size()) {
    // find first unclassified point:
    unsigned int idx = 0;
    for (unsigned int i = 0; i < takenCareOf.size(); ++i) {
      if (takenCareOf[i] == UNCLASSIFIED) {
	idx = i;
	break;
      }
    }
    // start new cluster with this point:
    clusters.push_back(zeroVec);
    unsigned int nowClus = clusters.size() - 1;
    clusters[nowClus].push_back(idx);
    takenCareOf[idx] = static_cast<int>(nowClus);
    ++pointsClassified;
    // cout << "Starting new cluster: " << idx << " " << v[idx] << " " << clusters.size() << endl;
    // now grow this cluster as much as possible
    bool solutionFound;
    do {
      solutionFound = false;
      for (unsigned int i = 0; i < takenCareOf.size(); ++i) {
	if (takenCareOf[i] != UNCLASSIFIED) {
	  continue;
	}
	for (unsigned int j = 0; j < clusters[nowClus].size(); ++j) {
	  if (vecDistanceSquare(v[clusters[nowClus][j]], v[i])
	      <= cutoffSqr) {
	    clusters[nowClus].push_back(i);
	    takenCareOf[i] = nowClus;
	    solutionFound = true;
	    ++pointsClassified;
	    break;
	  }
	}
      }
    } while (solutionFound);
  }
  /*
  for (unsigned int i = 0; i < v.size(); ++i) {
    // check if it fits into existing cluster:
    bool found = false;
    for (unsigned int j = 0; j < clusters.size(); ++j) {
      for (unsigned int k = 0; k < clusters[j].size(); ++k) {
	if (vecDistanceSquare(v[i], v[clusters[j][k]]) < cutoffSqr) {
	  clusters[j].push_back(i);
	  found = true;
	  break;
	}
      }
      if (found) {
	break;
      }
    }
    if (!found) {
      clusters.push_back(zeroVec);
      clusters[clusters.size() - 1].push_back(i);
    }
  }
  */

  // verify: sum of sizes of clusters must match number of input points:
  unsigned int sum = 0;
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    sum += clusters[i].size();
  }
  ERROR_IF(sum != v.size(), "Internal error in line 199!");
  return clusters;
}



/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > simpleRepresentativeLinkage(const Vec<Vector3D>& v, double cutoff)
{
  const int UNCLASSIFIED = -1;
  double cutoffSqr = cutoff * cutoff;
  Vec<Vec<unsigned int> > clusters;
  Vec<unsigned int> zeroVec;
  // cluster.push_back(zeroVec);
  // clusters[0].push_back(v[0]);
  Vec<int> takenCareOf(v.size(), UNCLASSIFIED);
  // takenCareOf[0] = 0; // point zero is put into cluster zero
  // unsigned int pointsClassified = 0; 
  for (unsigned int idx = 0; idx < v.size(); ++idx) {
    // find cluster which fits with representative:
    bool found = false;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      if (vecDistanceSquare(v[clusters[i][0]], v[idx]) <= cutoffSqr) {
	clusters[i].push_back(idx);
	takenCareOf[idx] = static_cast<int>(i);
	found = true;
	break;
      }
    }
    /* start new cluster with new representative */
    if (!found) {
      clusters.push_back(zeroVec);
      clusters[clusters.size() - 1].push_back(idx);
    }
  }
  // verify: sum of sizes of clusters must match number of input points:
  unsigned int sum = 0;
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    sum += clusters[i].size();
  }
  ERROR_IF(sum != v.size(), "Internal error in line 199!");
  return clusters;
}

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff */
Vec<Vec<unsigned int> > 
simpleRepresentativeLinkage(const Vec<Vec<double> >& v, double cutoff)
{
  const int UNCLASSIFIED = -1;
  double cutoffSqr = cutoff * cutoff;
  Vec<Vec<unsigned int> > clusters;
  Vec<unsigned int> zeroVec;
  // cluster.push_back(zeroVec);
  // clusters[0].push_back(v[0]);
  Vec<int> takenCareOf(v.size(), UNCLASSIFIED);
  // takenCareOf[0] = 0; // point zero is put into cluster zero
  // unsigned int pointsClassified = 0; 
  for (unsigned int idx = 0; idx < v.size(); ++idx) {
    // find cluster which fits with representative:
    bool found = false;
    for (unsigned int i = 0; i < clusters.size(); ++i) {
      if (euclidianDistanceSquare(v[clusters[i][0]], v[idx]) <= cutoffSqr) {
	clusters[i].push_back(idx);
	takenCareOf[idx] = static_cast<int>(i);
	found = true;
	break;
      }
    }
    if (!found) {
      clusters.push_back(zeroVec);
      clusters[clusters.size() - 1].push_back(idx);
    }
  }
  // verify: sum of sizes of clusters must match number of input points:
  unsigned int sum = 0;
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    sum += clusters[i].size();
  }
  ERROR_IF(sum != v.size(), "Internal error in line 199!");
  return clusters;
}

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff.
    It is fast because of presorting, however several representatives might be closer than cutoff
*/
Vec<Vec<unsigned int> > 
simpleSortLinkage(const Vec<Vec<double> >& vOrig, double cutoff)
{
  PRECOND(cutoff > 0.0);
  unsigned int nn = vOrig.size();
  unsigned int dim = vOrig[0].size();
  Vec<double> rOff(dim);
  Random& rnd = Random::getInstance();
  for (unsigned int i = 0; i < dim; ++i) {
    rOff[i] = rnd.getRandf(); // random offset, to avoid bias in rounding error
  }
  Vec<Vec<long> > v(nn, Vec<long>(dim)); // make working copy as "long"
  double mul = 1.0 / cutoff;
  for (unsigned int i = 0; i < nn; ++i) {
    for (unsigned int j = 0; j < dim; ++j) {
      v[i][j] = static_cast<long>((vOrig[i][j] * mul) + rOff[j]);
    }
  }
  Vec<Vec<unsigned int> > clusters;
  if (v.size() == 0) {
    return clusters;
  }
  Vec<unsigned int> order = documentedSort(v); // sort and store new order
  clusters.push_back(Vec<unsigned int>(1, order[0])); // store first element
  for (unsigned int idx = 1; idx < v.size(); ++idx) {
    if (v[clusters[clusters.size()-1][0]] == v[order[idx]]) {
      clusters[clusters.size()-1].push_back(order[idx]);
    }
    else {
      /* start new cluster with new representative */
      clusters.push_back(Vec<unsigned int>(1, order[idx]));
    }
  }
  // verify: sum of sizes of clusters must match number of input points:
//   unsigned int sum = 0;
//   for (unsigned int i = 0; i < clusters.size(); ++i) {
//     sum += clusters[i].size();
//   }
//   ERROR_IF(sum != v.size(), "Internal error in line 199!");
  return clusters;
}

/** 
    return vector with list of indices
    each list represents one class
    members of each class are closer than cutoff.
    It is fast because of presorting, however several representatives might be closer than cutoff
*/
/*
Vec<Vec<unsigned int> > 
simpleSortLinkage(const Vec<Vec<double> >& vOrig, double cutoff)
{
  Vec<Vec<double> > v = vOrig; // make working copy
  const int UNCLASSIFIED = -1;
  double cutoffSqr = cutoff * cutoff;
  Vec<Vec<unsigned int> > clusters;
  if (v.size() == 0) {
    return clusters;
  }
  Vec<unsigned int> order = documentedSort(v); // sort and store new order
  clusters.push_back(Vec<unsigned int>(1, order[0])); // store first element
  for (unsigned int idx = 1; idx < v.size(); ++idx) {
    if (euclidianDistanceSquare(v[clusters[clusters.size()-1][0]],
				v[order[idx]]) <= cutoffSqr) {
      clusters[clusters.size()-1].push_back(order[idx]);
    }
    else {
    clusters.push_back(Vec<unsigned int>(1, order[idx]));
    }
  }
  // verify: sum of sizes of clusters must match number of input points:
//   unsigned int sum = 0;
//   for (unsigned int i = 0; i < clusters.size(); ++i) {
//     sum += clusters[i].size();
//   }
//   ERROR_IF(sum != v.size(), "Internal error in line 199!");
  return clusters;
}
*/

/*

double
distToCluster(unsigned int v,
	    const Vec<unsigned int>& cluster,
	    const Vec<Vec<double> >& field)
{
  double result = 0.0;
  
  for (unsigned int i = 0; i < cluster.size(); ++i) {
    if (field[cluster[i]][v] > result) {
      result = field[cluster[i]][v];
    }
  }
  return result;
}

unsigned int
bestCluster(unsigned int v,
	    const Vec<Vec<unsigned int> > clusters,
	    const Vec<Vec<double> >& field, double cutoff)
{
  double bestDist = cutoff;
  unsigned int bestId = clusters.size();
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    double d = distToCluster(v, clusters[i], field);
    if (d < bestDist) {
      bestDist = d;
      bestId = i;
    }
  }
  return bestId;
}


Vec<Vec<unsigned int> > 
completeLinkage(const Vec<Vec<double> >& field,	double cutoff)
{
  
  Vec<Vec<unsigned int> > clusters;
  Vec<unsigned int> zeroVec;
  for (unsigned int i = 0; i < field.size(); ++i) {
    unsigned int best = bestCluster(i,clusters,field, cutoff);
    if (best >= clusters.size()) {
      clusters.push_back(zeroVec);
      clusters[clusters.size()-1].push_back(i);
    }
    else {
      clusters[best].push_back(i);
    }
  }

  return clusters;
}
*/

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  unsigned int k)
{
  // cout << "Starting knn!" << endl;
  if (k < 1) {
    return Vec<unsigned int>();
  }
  if (k >= data.size()) {
    return generateStair(data.size());
  }
  Vec<unsigned int> result(k);
  Vec<double> distances(k);
  // cout << "Hi1!" << endl;
  for (unsigned int i = 0; i < k; ++i) {
    result[i] = i;
    distances[i] = euclidianDistanceSquare(query, data[i]);
  }
  // cout << "Hi2!" << endl;
  parallelSort(distances, result);
  double worstDist = distances[distances.size()-1];
  double d;
  // cout << "Hi3!" << endl;
  for (unsigned int i = k; i < data.size(); ++i) {
    d = euclidianDistanceSquare(query, data[i]);
    if (d >= worstDist) {
      continue;
    }
    // insert it in list:
    // cout << "Hi4!" << endl;
    for (unsigned int j = 0; j < k; ++j) {
      INVARIANT(distances.size() == k);
      INVARIANT(result.size() == k);
      if (d < distances[j]) {
	distances.insert(distances.begin()+j, d);
	result.insert(result.begin()+j, i);
	distances.erase(distances.begin()+distances.size()-1);
	result.erase(result.begin()+result.size()-1);
	worstDist = distances[distances.size()-1];
	break;
      }
    }
  }
  // cout << "Ending knn!" << endl;
  return result;
}

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  unsigned int k, 
		  const Vec<double>& scaling)
{
  unsigned int maxSize = 5 * k; // maximum size of result
  // cout << "Starting knn!" << endl;
  if (k < 1) {
    return Vec<unsigned int>();
  }
  if (k >= data.size()) {
    return generateStair(data.size());
  }
  Vec<unsigned int> result(k);
  Vec<double> distances(k);
  Vec<double> allDist(data.size(), 0.0); // store all distances
  for (unsigned int i = 0; i < k; ++i) {
    result[i] = i;
    distances[i] = euclidianDistanceSquare(query, data[i], scaling);
  }
  parallelSort(distances, result);
  double worstDist = distances[distances.size()-1];
  double d;

  for (unsigned int i = 0; i < k; ++i) {
    allDist[result[i]] =  euclidianDistanceSquare(query, data[result[i]], 
						  scaling);
  }

  for (unsigned int i = k; i < data.size(); ++i) {
    d = euclidianDistanceSquare(query, data[i], scaling);
    allDist[i] = d;
    if (d > worstDist) {
      continue;
    }
    // insert it in list:
    for (unsigned int j = 0; j < k; ++j) {
      INVARIANT(distances.size() == k);
      INVARIANT(result.size() == k);
      if (d < distances[j]) {
	distances.insert(distances.begin()+j, d);
	  result.insert(result.begin()+j, i);
	  distances.erase(distances.begin()+distances.size()-1);
	  result.erase(result.begin()+result.size()-1);
	  worstDist = distances[distances.size()-1];
	  break;
      }
    }
  }

  // postprocessing: possibly add equal members:
  for (unsigned int i = 0; i < data.size(); ++i) {
    if (allDist[i] == worstDist) {
      bool found = false;
      // check if not already part of solution:
      for (unsigned int j = 0; j < result.size(); ++j) {
	if (result[j] == i) {
	  found = true;
	  break;
	}
      }
      if (!found) {
	distances.push_back(allDist[i]);
	result.push_back(i);
	if (result.size() >= maxSize) {
	  break;
	}
      }
    }
  }

  // cout << "Ending knn!" << endl;
  return result;
}

/*
Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  unsigned int k, 
		  const Vec<double>& scaling,
		  const Vec<Vec<unsigned int> >& clusters,
		  double clusterCutoff)
{
  // cout << "Starting knn!" << endl;
  if (k < 1) {
    return Vec<unsigned int>();
  }
  else if (k >= data.size()) {
    return generateStair(data.size());
  }
  else if (k > clusters.size()) {
    return kNearestNeighbors(query, data, k, scaling); // conventional method if not enough clusters
  }
  // find k nearest clusters:
  Vec<double> distances(clusters.size());
  Vec<unsigned int> result(clusters.size()); // store data element ids
  Vec<unsigned int> clusterIds(clusters.size());

  for (unsigned int i = 0; i < clusters.size(); ++i) {
    clusterIds[i] = i;
    distances[i] = euclidianDistanceSquare(query, data[clusters[i][0]], scaling);
  }
  parallelSort(distances, clusterIds);
  if (clusterIds.size() > k) {
    distances.erase(distances.begin()+k, distances.end());
    clusterIds.erase(clusterIds.begin()+k, clusterIds.end());
    result.erase(result.begin()+k, result.end());
  }
  for (unsigned int i = 0; i < result.size(); ++i) {
    result[i] = clusters[clusterIds[i]][0];
  }
  ERROR_IF(distances.size() > k, "Internal error in line 477!");
  ERROR_IF(clusterIds.size() > k, "Internal error in line 478!");

  double worstDist = distances[distances.size()-1];
  double d;
  unsigned int dIdx; // data index  
  // loop over k nearest clusters:
  for (unsigned int n = 0; n < k; ++n) {
    unsigned int cl = clusterIds[n];
    for (unsigned int i = 0; i < clusters[cl].size(); ++i) {
      dIdx = clusters[cl][i]; // data element
      d = euclidianDistanceSquare(query, data[dIdx], scaling);
      if (d >= worstDist) {
	continue;
      }
      // insert it in list:
      if (distances.size() == k) {
	for (unsigned int j = 0; j < k; ++j) {
	  INVARIANT(distances.size() == k);
	  INVARIANT(result.size() == k);
	  if (d < distances[j]) {
	    distances.insert(distances.begin()+j, d);
	    result.insert(result.begin()+j, dIdx);
	    distances.erase(distances.begin()+distances.size()-1);
	    result.erase(result.begin()+result.size()-1);
	    worstDist = distances[distances.size()-1];
	    break;
	  }
	}
      }
      else { // still grow list
	ASSERT(distances.size() < k);
	for (unsigned int j = 0; j < k; ++j) {
	  if (d < distances[j]) {
	    distances.insert(distances.begin()+j, d);
	    result.insert(result.begin()+j, dIdx);
	    worstDist = distances[distances.size()-1];
	    break;
	  }
	}
      }
    }
  }
  // cout << "Ending knn!" << endl;
  return result;

}
*/

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  unsigned int k, 
		  const Vec<double>& scaling,
		  const Vec<Vec<unsigned int> >& clusters,
		  double clusterCutoff)
{
  double clusterCutoffOrig = clusterCutoff;
  clusterCutoff *= clusterCutoff;
    // cout << "Starting knn!" << endl;
  if (k < 1) {
    return Vec<unsigned int>();
  }
  else if (k >= data.size()) {
    return generateStair(data.size());
  }
//   else if (k > clusters.size()) {
//     return kNearestNeighbors(query, data, k, scaling); // conventional method if not enough clusters
//   }
  PurgePriorityQueue<RankedSolution3<unsigned int> > queue;
  queue.setMaxSize(3 * k);
  // find nearest cluster:
  Vec<double> distances(clusters.size());

  Vec<unsigned int> clusterIds(clusters.size());
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    clusterIds[i] = i;
    distances[i] = euclidianDistanceSquare(query, data[clusters[i][0]], scaling);
  }
  parallelSort(distances, clusterIds);
  // double dClusterBest = euclidianDistanceSquare(query, data[clusters[clusterIds[0]][0]], scaling);
  double d;
  unsigned int cl; //  = bestClusterId;
  unsigned int dIdx;
  RankedSolution3<unsigned int> solution;

  for (unsigned int n = 0; n < clusterIds.size(); ++n) {
    cl = clusterIds[n];
    if (queue.hasLeaked()) {
	if ((euclidianDistance(query, data[clusters[cl][0]]) - clusterCutoffOrig) > (-(queue.getBestPurged().first))) {
	  continue; // too far away, cannot contain good members
	}
     }
    for (unsigned int i = 0; i < clusters[cl].size(); ++i) {
      dIdx = clusters[cl][i]; // data element
      d = euclidianDistanceSquare(query, data[dIdx], scaling);
      solution.first = -d;
      solution.second = dIdx;
      queue.push(solution);

    }
  }

  unsigned int minDim = k;
  if (queue.size() < k) {
    minDim = queue.size();
  }
  Vec<unsigned int> result(minDim); // (clusters.size()); // store data element ids
  //  cout << "Query was: " << query << endl;
  for (unsigned int i = 0; i < result.size(); ++i) {
    result[i] = queue.top().second;
//     cout << i + 1 << " " << queue.top().first << " " << queue.top().second << "  "
// 	 << data[queue.top().second] << endl;
    queue.pop();
  }
  // cout << "Ending knn!" << endl;
  ERROR_IF((data.size() > k) && (result.size() < k),
	   "Internal error in line 696!");
  return result;

}

Vec<unsigned int>
kNearestNeighbors(const Vec<double>& query,
		  const Vec<Vec<double> >& data,
		  const Vec<Vec<Vec<double> > >& clustData,
		  unsigned int k, 
		  const Vec<double>& scaling,
		  const Vec<Vec<unsigned int> >& clusters,
		  const Vec<Vec<Vec<unsigned int> > >& subClusters,
		  double clusterCutoff,
		  double clusterCutoff2)
{
  double clusterCutoffOrig = clusterCutoff;
  clusterCutoff *= clusterCutoff;
    // cout << "Starting knn!" << endl;
  if (k < 1) {
    return Vec<unsigned int>();
  }
  else if (k >= data.size()) {
    return generateStair(data.size());
  }
//   else if (k > clusters.size()) {
//     return kNearestNeighbors(query, data, k, scaling); // conventional method if not enough clusters
//   }
  PurgePriorityQueue<RankedSolution3<unsigned int> > queue;
  queue.setMaxSize(3 * k);
  // find nearest cluster:
  Vec<double> distances(clusters.size());

  Vec<unsigned int> clusterIds(clusters.size());
  for (unsigned int i = 0; i < clusters.size(); ++i) {
    clusterIds[i] = i;
    distances[i] = euclidianDistanceSquare(query, data[clusters[i][0]], scaling);
  }
  parallelSort(distances, clusterIds);
  // double dClusterBest = euclidianDistanceSquare(query, data[clusters[clusterIds[0]][0]], scaling);
  double d;
  unsigned int cl; //  = bestClusterId;
  unsigned int dIdx;
  RankedSolution3<unsigned int> solution;

  Vec<unsigned int> tmpResult;
  for (unsigned int n = 0; n < clusterIds.size(); ++n) {
    cl = clusterIds[n];
    if (queue.hasLeaked()) {
      if ((euclidianDistance(query, data[clusters[cl][0]]) 
	   - clusterCutoffOrig) > (-(queue.getBestPurged().first))) {
	  continue; // too far away, cannot contain good members
	}
     }
    tmpResult = kNearestNeighbors(query, clustData[cl], k, scaling, 
				  subClusters[cl], clusterCutoff2);
    for (unsigned int i = 0; i < tmpResult.size(); ++i) {
      dIdx = clusters[cl][tmpResult[i]]; // data element
      d = euclidianDistanceSquare(query, data[dIdx], scaling);
      solution.first = -d;
      solution.second = dIdx;
      queue.push(solution);
      if (!queue.isAccepted()) {
	break; // all further solutions are only worse
      }
    }
  }

  unsigned int minDim = k;
  if (queue.size() < k) {
    minDim = queue.size();
  }
  Vec<unsigned int> result(minDim); // (clusters.size()); // store data element ids
  //  cout << "Query was: " << query << endl;
  for (unsigned int i = 0; i < result.size(); ++i) {
    result[i] = queue.top().second;
//     cout << i + 1 << " " << queue.top().first << " " << queue.top().second << "  "
// 	 << data[queue.top().second] << endl;
    queue.pop();
  }
  // cout << "Ending knn!" << endl;
  ERROR_IF((data.size() > k) && (result.size() < k),
	   "Internal error in line 696!");
  return result;

}



Vec<unsigned int>
knnThin(Vec<Vec<double> > data,
	Vec<int> classes,
	unsigned int k)
{
  // cout << "Starting knnThin!" << endl;
  Vec<unsigned int> resultIndices = generateStair(data.size());
  for (int i = data.size()-1; (i >= 0) && (data.size() > k); --i) {
    Vec<unsigned int> result = kNearestNeighbors(data[i], data, k);
#ifdef DEBUG_VERBOSE
    cout << "# Working on data vector : " << data.size() << "  " << i 
	 << " knn result: ";
    for (unsigned int j = 0; j < result.size(); ++j) {
      cout << result[j] << " " << classes[result[j]] << "  ";
    }
    cout << data[i];
#endif
    // check if classes diverse:
    int firstClass = classes[i];
    bool found = false;
    for (unsigned int j = 0; j < result.size(); ++j) {
      if (classes[result[j]] != firstClass) {
	found = true;
	break;
      }
    }
    if (!found) { // all neighbors belong to same class:
      // cout << "# Deleting data point " << i + 1 << endl;
      data.erase(data.begin()+i);
      classes.erase(classes.begin()+i);
      resultIndices.erase(resultIndices.begin()+i);
//       cout << "Remaining data: " << data << " classes: " 
// 	   << classes << endl;
    }
  }
  // cout << "Final indices: " << resultIndices << endl;
  return resultIndices;
}


Vec<unsigned int>
knnThin(Vec<Vec<double> > data,
	Vec<unsigned int> classes,
	unsigned int k)
{
  // cout << "Starting knnThin!" << endl;
  Vec<unsigned int> resultIndices = generateStair(data.size());
  for (int i = data.size()-1; (i >= 0) && (data.size() > k); --i) {
    Vec<unsigned int> result = kNearestNeighbors(data[i], data, k);
#ifdef DEBUG_VERBOSE
    cout << "# Working on data vector : " << data.size() << "  " << i 
	 << " knn result: ";
    for (unsigned int j = 0; j < result.size(); ++j) {
      cout << result[j] << " " << classes[result[j]] << "  ";
    }
    cout << data[i];
#endif
    // check if classes diverse:
    unsigned int firstClass = classes[i];
    bool found = false;
    for (unsigned int j = 0; j < result.size(); ++j) {
      if (classes[result[j]] != firstClass) {
	found = true;
	break;
      }
    }
    if (!found) { // all neighbors belong to same class:
      // cout << "# Deleting data point " << i + 1 << endl;
      data.erase(data.begin()+i);
      classes.erase(classes.begin()+i);
      resultIndices.erase(resultIndices.begin()+i);
//       cout << "Remaining data: " << data << " classes: " 
// 	   << classes << endl;
    }
  }
  // cout << "Final indices: " << resultIndices << endl;
  return resultIndices;
}

/** returns true if two datavectors are more similar (abs norm) than cutoff */
bool
isDataVecSimilar(const Vec<double>& dataVec,
		 const Vec<double>& lastVec, 
		 const Vec<unsigned int>& mask, 
		 double cutoff)
{
  if (dataVec.size() != lastVec.size()) {
    return false;
  }
  double sum = 0.0;
  for (unsigned int i = 0; i < mask.size(); ++i) {
    sum += fabs(dataVec[mask[i]]-lastVec[mask[i]]);
    if (sum > cutoff) {
      return false;
    }
  }
  return true;
}

Vec<unsigned int>
uniquify(const Vec<Vec<double> >& data,
	 Vec<unsigned int> mask,
	 unsigned int simMax,
	 double cutoff) 
{
  // Vec<double> dataVec, lastVec;
  unsigned int lastVecId = 0;
  Vec<unsigned int> lastLines;
  Vec<unsigned int> ranIndices;
  Vec<unsigned int> result;
  for (unsigned int k = 0; k < data.size(); ++k) {
    // 	 << " last vec: " << lastVec << endl;
    if ((lastLines.size() != 0) && (!isDataVecSimilar(data[k], data[lastVecId], mask, cutoff))) {
      // lastLines.push_back(line);
      unsigned int mini = simMax;
      if (lastLines.size() < simMax) {
	mini = lastLines.size();
      }
      // output of random sample
      ranIndices = generateRandomIndexSubset(mini, lastLines.size(), 0);
      for (unsigned int j = 0; j < ranIndices.size(); ++j) {
	ASSERT(k >= ranIndices[j]);
	result.push_back(lastLines[ranIndices[j]]);
      }
      lastLines.clear();
    }
    lastLines.push_back(k);
    lastVecId = k;
  }
  // if buffer still full at end of file:
  if (lastLines.size() > 0) {
    unsigned int mini = simMax;
    if (lastLines.size() < simMax) {
      mini = lastLines.size();
    }
    // output of random sample
    ranIndices = generateRandomIndexSubset(mini, lastLines.size(), 0);
    for (unsigned int j = 0; j < ranIndices.size(); ++j) {
      result.push_back(lastLines[ranIndices[j]]);
    }
    lastLines.clear();
  }
  return result;
}
