// --*- C++ -*------x---------------------------------------------------------
// $Id:
//
// Class:           IntervallIntSet
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     Implements concepts of intervall.
//                  Upper and lower bounds are integers.
// 
// Reviewed by:     -
// -----------------x-------------------x-------------------x-----------------

#ifndef __INTERVALL_INT_SET_H__
#define __INTERVALL_INT_SET_H__

// Includes

#include <iostream>

#include <debug.h>
#include <vectornumerics.h>
#include <IntervallInt.h>
#include <algorithm>

using namespace std;

/** Set storing instances of an intervall class, which stores upper and lower limits (both are inclusive)
 * The set automatically consolidates overlapping intervalls.
 *  @author Eckart Bindewald
 *  @see    -
 *  @review - 
 */
class IntervallIntSet {
public:

  /* TYPEDEFS */

  typedef IntervallInt::index_type index_type;
  typedef Vec<IntervallInt> set_type;
  typedef Vec<IntervallInt>::size_type size_type;

  /* CONSTRUCTORS */

  IntervallIntSet() : lower(0), upper(0) { }

  IntervallIntSet(const IntervallIntSet& orig) { copy(orig); }

  virtual ~IntervallIntSet() { }

  /* OPERATORS */

  /** Assigment operator. */
  IntervallIntSet& operator = (const IntervallIntSet& orig) {
    if (&orig != this) {
      copy(orig);
    }
    return *this;
  } 

  friend ostream& operator << (ostream& os, const IntervallIntSet& rval);
  /*
    friend istream& operator >> (istream& is, Edge& rval);
  */

  /* PREDICATES */

  /** Adds padding to all intervalls, avoiding overlaps and collisions */
  void addPadding(index_type padding, index_type indexMin, index_type indexMax);

  const IntervallInt& getIntervall(size_type n) const { return intervalls[n]; }

  const IntervallInt& operator [](size_type n) const { return intervalls[n]; }

  /** returns combined length of all intervalls */
  int getLengthCount() {
    int sum = 0;
    for (size_type i = 0; i < intervalls.size(); ++i) {
      sum += intervalls[i].getLength();
    }
    return sum;
  }

  int getLower() const { return lower; }

  int getUpper() const { return upper; }

  /** returns true given intervall overlapps with one position */
  size_type isOverlapping(index_type index) const {
    if ((index < lower)  || (index > upper)) {
      return size();
    }
    IntervallInt searchIntervall(index, index);
    set_type::const_iterator it = lower_bound(intervalls.begin(), intervalls.end(), searchIntervall);
    if (it != intervalls.end()) {
      if ((*it).isOverlapping(searchIntervall)) {
	return (distance(intervalls.begin(), it));
      } else if ((it != intervalls.begin()) && (*(it-1)).isOverlapping(searchIntervall)) {
	return (distance(intervalls.begin(), (it-1)));
      }
    }
    return size();
  }

  /** returns true given intervall overlapps with one intervall in solution */
  bool isOverlapping(const IntervallInt& searchIntervall) const {
    index_type index = searchIntervall.getLower();
    index_type index2 = searchIntervall.getUpper();
    bool result = false;
    if ((index2 < lower)  || (index > upper)) {
      result = false;
    } else {
      set_type::const_iterator it = lower_bound(intervalls.begin(), intervalls.end(), searchIntervall);
      if (it != intervalls.end()) {
	if ((*it).isOverlapping(searchIntervall)) {
	  result = true; // (distance(intervalls.begin(), it));
	} else if ((it != intervalls.begin()) && (*(it-1)).isOverlapping(searchIntervall)) {
	  result = true; // (distance(intervalls.begin(), it));
	}
      }
    }
    ASSERT(result == isOverlappingSafe(searchIntervall));
    return result; // size();
  }


  /** returns true given intervall overlapps with one intervall in solution */
  bool isOverlappingSafe(const IntervallInt& intervall) const {
    for (size_type i = 0; i < intervalls.size(); ++i) {
      if (intervalls[i].isOverlapping(intervall)) {
	return true;
      }
    }
    return false;
  }

  /** returns true given intervall overlapps with one intervall in solution */
  bool isOverlapping(const IntervallIntSet& otherSet) const {
    for (size_type i = 0; i < otherSet.size(); ++i) {
      if (isOverlapping(otherSet.getIntervall(i))) {
	return true;
      }
    }
    return false;
  }

  size_type size() const { return intervalls.size(); }

  /* MODIFIERS */

  void add(const IntervallInt& intervall, bool consolidateMode=true) {
    bool overlapFlag = false;
    if ((intervalls.size() == 0) || (intervall.getLower() < lower)) {
      lower = intervall.getLower();
    }
    if ((intervalls.size() == 0) || (intervall.getUpper() > upper)) {
      upper = intervall.getUpper();
    }
    set_type::iterator it = lower_bound(intervalls.begin(), intervalls.end(), intervall);
    if ((it != intervalls.begin()) && ((*(it-1)).isOverlapping(intervall))) {
	overlapFlag = true;
	(*(it-1)).consolidate(intervall);
    } else if ((it != intervalls.end()) && ((*it).isOverlapping(intervall) ) ) {
      overlapFlag = true;
      (*it).consolidate(intervall);
    } else {
      intervalls.insert(it, intervall);
    }
    if (overlapFlag && consolidateMode) {
      consolidate();
    }
  }

  void clear() {
    lower = 0;
    upper = 0;
    intervalls.clear();
  }

  void merge(const IntervallIntSet& other) {
    for (size_type i = 0; i < other.size(); ++i) {
      add(other.getIntervall(i));
    }
  }

  /** Modify intervalls so that they are not overlapping (leaves unchanged or reduces size of this object) */
  void consolidate();

  /** Reserves sufficient size */
  void reserve(size_type sz) {
    intervalls.reserve(sz);
  }

protected:
  /* OPERATORS  */

  /* PREDICATES */

  /* MODIFIERS  */

  void copy(const IntervallIntSet& other) { 
    lower = other.lower;
    upper = other.upper;
    intervalls = other.intervalls;
  }

private:

  /* OPERATORS  */

  /* PREDICATES */

  /* MODIFIERS  */

  /** Adds padding to all intervalls, avoiding overlaps and collisions.
      Index min / max are inclusive intervall descriptors for maximum and minium values of interval bounds. */
  void addPadding(index_type padding, index_type indexMin, index_type indexMax, 
		  size_type n, index_type prevUpper, index_type nextLower);



private:

  /* PRIVATE ATTRIBUTES */

  int lower;
  
  int upper;

  Vec<IntervallInt> intervalls; // *Sorted* vector of intervalls

};


inline
bool operator < (const IntervallIntSet& intervallA, 
		 const IntervallIntSet& intervallB)
{
  return (intervallA.getLower() < intervallB.getLower());
}

inline
ostream& operator << (ostream& os, const IntervallIntSet& rval)
{
  os << rval.size() << "  ";
  for (IntervallIntSet::size_type i = 0; i < rval.size(); ++i) {
    os << rval.getIntervall(i) << " ";
  }
  return os;
}
#endif /* __EDGE_H__ */

