// --*- C++ -*------x---------------------------------------------------------
// $Id: CoordinateSystem.cc,v 1.2 2006/07/03 16:20:48 bindewae Exp $
//
// Class:           CoordinateSystem
// 
// Base class:      -
//
// Derived classes: - 
//
// Author:          Eckart Bindewald
//
// Description:     missing
// 
// -----------------x-------------------x-------------------x-----------------

#include <CoordinateSystem.h>
#include <vectornumerics.h>

// ---------------------------------------------------------------------------
//                                   CoordinateSystem
// -----------------x-------------------x-------------------x-----------------

/* CONSTRUCTORS */

/** default operator
  */
CoordinateSystem::CoordinateSystem() : origin(0.0,0.0,0.0),
	       ex(1.0,0.0,0.0),ey(0.0,1.0,0.0),ez(0.0,0.0,1.0)
{
  updateMatrix();
  // no precondition (except sufficient memory)
  // postcondition depends on class
}

CoordinateSystem::CoordinateSystem(const Vector3D& _origin,
		   const Vector3D& _x, const Vector3D& _y)
{
  setOrigin(_origin);
  setAxis(_x, _y);
}


/** copy constructor
  */
CoordinateSystem::CoordinateSystem(const CoordinateSystem& orig) :
  origin(orig.origin),ex(orig.ex),ey(orig.ey),ez(orig.ez),
  m(orig.m), mt(orig.mt)
{
  // no precondition (except sufficient memory)
  //  POSTCOND( (orig == *this));
}

/** destructor
  */
CoordinateSystem::~CoordinateSystem()
{
  // no precondition (except sufficient memory)
  // postcondition depends on class
}


/* OPERATORS */

/** Assignement operator. */
CoordinateSystem& CoordinateSystem::operator = (const CoordinateSystem& orig)
{
  // no precondition
  if (&orig != this)
    {
      copy(orig);
    }
  // POSTCOND( (orig == *this));
  return *this;
}

/** comparison operator. 
*/
bool 
operator == (const CoordinateSystem& left,const CoordinateSystem& right)
{
  // no precondition
  return left.compare(right);
}

/** output operator 
*/
ostream& 
operator << (ostream& os, const CoordinateSystem& r)
{
  PRECOND(os);

  os << r.origin << " " << r.ex << " " << r.ey;

  POSTCOND(os);
  return os;
}

/** input operator  
*/
istream& 
operator >> (istream& is, CoordinateSystem& r)
{
  PRECOND(is);
  Vector3D v1,v2,v3;
  is >> v1 >> v2 >> v3;
  r.setOrigin(v1);
  r.setAxis(v2,v3);
  POSTCOND(is);
  return is;
}


/* PREDICATES */


/** return true, if all members of other and this object 
   give true with respect to "==" operator.
  */
bool 
CoordinateSystem::compare(const CoordinateSystem& other) const
{
  return (origin == other.origin)
    && (ex == other.ex) && (ey == other.ey) && (ez == other.ez);
}

/** return position with respect to THIS coordinate system */
/* 
Vector3D 
CoordinateSystem::newCoordinates(const Vector3D& v) const
{
  PRECOND(isProperlyDefined(),  exception);
#ifndef NDEBUG
  // cout <<  " matrix before rotation: " << m << endl;
  Vector3D v2 = v - origin;
  Vector3D vresult = m * v2;
  // check if rotation did not change length of vector
  if (fabs(v2.length() - vresult.length()) > 0.1) {
    cout << "Error in rotation!" << endl;
    cout << "Matrix: " << *this << endl;
    cout << "original vector: " << v << endl;
    cout << "translated original vector: " << v2 << "  " 
	 << v2.length() << endl;
    cout << "rotated vector: " << vresult << "  " 
	 << vresult.length() << endl;
    ERROR("Ill defined coordinate system!" );
  }
#endif  
  return m * (v - origin);
}
*/

/** return position with respect to global coordinate system */
/*
Vector3D 
CoordinateSystem::oldCoordinates(const Vector3D& v) const
{
  PRECOND(isProperlyDefined() && (v.length() < 10000.0) );
#ifndef NDEBUG
  Vector3D result = (mt * v) + origin;
  if (result.length() > 10000.0) {
    cout << v << endl << result << endl << *this << endl;
  }
#endif
  //   POSTCOND(result.length() < 10000.0);
  return (mt * v) + origin;
}
*/

/** return true, if determinante of direction vectors is 1.0 */
bool 
CoordinateSystem::isProperlyDefined() const
{
  bool result = true;
  // also make sure that angles are rectangular:
  result = result && (fabs(ex * ey) < 0.01);
  result = result && (fabs(ex.length() - 1.0) < 0.01)
    && (fabs(ey.length() - 1.0) < 0.01);
//   if (result) {
//     Vec<double> vdrow(3);
//     Vec<Vec<double> > vdmatrix3(3,vdrow);
//     vdmatrix3[0][0] = ex.x();
//     vdmatrix3[1][0] = ey.x();
//     vdmatrix3[2][0] = ez.x();
    
//     vdmatrix3[0][1] = ex.y();
//     vdmatrix3[1][1] = ey.y();
//     vdmatrix3[2][1] = ez.y();
    
//     vdmatrix3[0][2] = ex.z();
//     vdmatrix3[1][2] = ey.z();
//     vdmatrix3[2][2] = ez.z();
    
//     double det = matrixDeterminante(vdmatrix3);
//     result = result 
//       && (fabs(1.0 - det) < 0.01); // true, if determinante close to 1
//   }
  if (!result) {
    cout <<  "Coordinate system not properly defined!"
	 << *this << endl;
  }
  return result;
}


/* MODIFIERS */

/** copies orig object to this object ("deep copy")
  */
void 
CoordinateSystem::copy(const CoordinateSystem& orig)
{
  // no precondition
  origin = orig.origin;
  ex = orig.ex;
  ey = orig.ey;
  ez = orig.ez;
  m  = orig.m;
  mt = orig.mt;
  //  POSTCOND(orig == *this);
}

/** set x, y and z axis of coordinate system */
void 
CoordinateSystem::setAxis(const Vector3D& v1, const Vector3D& v2) 
{ 
//    cout << "setting coordinate system with axis: "
//         << v1 << " " << v2 << endl;
  ex = v1; 
  ex.normalize(); 
  ey = v2; 
  ey.normalize(); 
  ey = ey - ex * (ey * ex); // make rectengular  
  ey.normalize();
  ASSERT(fabs(ex * ey) < 0.01);
  ASSERT((fabs(ex.length() - 1.0) < 0.01)
	 && (fabs(ey.length() - 1.0) < 0.01));
  ez = cross(ex,ey); 
  updateMatrix();
  //  cout << "result after setting: " << *this << endl; 
  POSTCOND(isProperlyDefined());
}

/** apply Euler angle rotation to axis of coordinate system
    does not change origin!
    angles are in radians. */
void 
CoordinateSystem::rotateEulerAngles(double u, double v, double w)
{
  PRECOND(isProperlyDefined());
  Matrix3D euler = computeEulerRotationMatrix(u,v,w);
  Vector3D extmp = euler * ex;
  Vector3D eytmp = euler * ey;
  // make sure it was a rotation that did not change lengths:
//    cout << "Vectors before rotation: " << ex << " " <<  ey << endl;
//    cout << "Vectors after rotation: " << extmp << " " <<  eytmp << endl;
//    cout << "Rotation matrix:" << euler << endl;
  ASSERT( (fabs(extmp.length()-1.0) <= 0.01)
	    || (fabs(eytmp.length()-1.0) > 0.01));
  ASSERT( (fabs(extmp * eytmp) <= 0.01));
  ASSERT( (fabs(extmp.length()-ex.length()) <= 0.01)
	    || (fabs(eytmp.length()-ey.length()) > 0.01));
  setAxis(extmp, eytmp);
  POSTCOND(isProperlyDefined());
}
