#include <MAFAlignmentTools.h>

/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
 */
int
MAFAlignmentTools::stemExpandability1Fast(const MAFAlignment& maf, const MAFAlignment& maf2, 
					  const Stem& stem, bool guAllowed, bool complement) {
  PRECOND(complement); // take out later
  string methodName = "stemExpandability1Fast";
  REMARK << "Warning: fast version of " << methodName << " not tested yet."
	 << endl;
  ASSERT(maf.getRefAssembly() == maf2.getRefAssembly());
  // REMARK << "Starting " << methodName << endl;
  int result = 0;
  Stem::index_type firstStartAssem = stem.getStart();
  Stem::index_type firstStopAssem = stem.getStop();
  // Returns the internal column id for a given position in assembly coordinates (using reference assembly). 
  length_type firstStartCol = maf.convertAssemblyPositionToColumnId(firstStartAssem);
  length_type firstStopCol = maf2.convertAssemblyPositionToColumnId(firstStopAssem);
  length_type firstStartAliId = 0;
  length_type firstStartAliColId = 0;
  length_type firstStopAliId = 0;
  length_type firstStopAliColId = 0;
  maf.findAlignmentIdColumnId(firstStartCol, &firstStartAliId, &firstStartAliColId);
  maf2.findAlignmentIdColumnId(firstStopCol, &firstStopAliId, &firstStopAliColId);
  // REMARK << "Converted start column: " << firstStartAssem << " " << firstStartCol << " " << firstStartAliId << " " << firstStartAliColId << endl; 
  // REMARK << "Converted stop column: " << firstStopAssem << " " << firstStopCol << " " << firstStopAliId << " " << firstStopAliColId << endl; 
  SequenceAlignment::size_type sid1 = maf.getAssemblyRowId(firstStartAliId, maf.getRefAssembly());
  SequenceAlignment::size_type sid2 = maf2.getAssemblyRowId(firstStopAliId, maf.getRefAssembly());
  ASSERT(sid1 < maf[firstStartAliId].size());
  ASSERT(sid2 < maf2[firstStopAliId].size());
  string sequence1 = maf[firstStartAliId].getSequence(sid1); // get ref sequence from MAF block
  string sequence2 = maf2[firstStopAliId].getSequence(sid2); // get ref sequence from MAF block
  // find out which alignment block and which column 
  for (result = 0; result < stem.getStart(); ++result) {
    length_type startSeqPos = firstStartAliColId - result;
    if (startSeqPos < 0) {
      if (firstStartAliId == 0) {
	break; // no smaller MAF blocks
      }
      break; // avoid switching MAF blocks at the moment
    }
    length_type stopSeqPos  = firstStopAliColId + result;
    if (stopSeqPos >= static_cast<length_type>(sequence2.size())) {
      break; // avoid switching MAF blocks at the moment
    }
    char c1 = sequence1[startSeqPos];
    char c2 = sequence2[stopSeqPos];
    if (complement) {
      c1 = RnaSecondaryStructureTools::generateComplementDna(c1);
      c2 = RnaSecondaryStructureTools::generateComplementDna(c2);
    }
    // bool isComplement = RnaSecondaryStructureTools::isWatsonCrick(c1, c2, guAllowed);
    bool isComplement = NucleotideTools::isComplementary(c1, c2, guAllowed, false);
    if (!isComplement) {
      if (result == 0) {
	cout << "Stem " << stem << " contains non-Watson-Crick base pairs." << endl;
	ASSERT(false);
      }
      break;
    }
  }
  --result;
  ASSERT(result >= 0);
  // REMARK << "Finished " << methodName << endl;
  ASSERT(complement);
  ASSERT(result == stemExpandability1Slow(maf,maf2,stem, guAllowed, complement)); // has to match slower and tested version
  return result;
} 

/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
 */
int
MAFAlignmentTools::stemExpandability2Fast(const MAFAlignment& maf, const MAFAlignment& maf2, 
				const Stem& stem, bool guAllowed, bool complement) {
    ASSERT(maf.getRefAssembly() == maf2.getRefAssembly());
    string methodName = "stemExpandability2Fast";
    // REMARK << "Starting " << methodName << endl;
    int result = 0;
    Stem::index_type firstStart = stem.getStart();
    Stem::index_type firstStop = stem.getStop();
    // Returns the internal column id for a given position in assembly coordinates (using reference assembly). 
    length_type firstStartCol = maf.convertAssemblyPositionToColumnId(firstStart);
    length_type firstStopCol = maf2.convertAssemblyPositionToColumnId(firstStop);
    length_type firstStartAliId = 0;
    length_type firstStartAliColId = 0;
    length_type firstStopAliId = 0;
    length_type firstStopAliColId = 0;
    maf.findAlignmentIdColumnId(firstStartCol, &firstStartAliId, &firstStartAliColId);
    maf2.findAlignmentIdColumnId(firstStopCol, &firstStopAliId, &firstStopAliColId);
    SequenceAlignment::size_type sid1 = maf.getAssemblyRowId(firstStartAliId, maf.getRefAssembly());
    SequenceAlignment::size_type sid2 = maf2.getAssemblyRowId(firstStopAliId, maf2.getRefAssembly());
    ASSERT(sid1 < maf[firstStartAliId].size());
    ASSERT(sid2 < maf2[firstStopAliId].size());
    string sequence1 = maf[firstStartAliId].getSequence(sid1); // get ref sequence from MAF block
    string sequence2 = maf2[firstStopAliId].getSequence(sid2); // get ref sequence from MAF block
    // find out which alignment block and which column 
    for (result = 0; result < stem.getStart(); ++result) {
      Stem::index_type newLength = stem.getLength() + result;
      length_type startSeqPos = firstStartAliColId + newLength - 1;
      if (startSeqPos < 0) {
	if (firstStartAliId == 0) {
	  break; // no smaller MAF blocks
	}
	break; // avoid switching MAF blocks at the moment
      }
      length_type stopSeqPos  = firstStopAliColId - newLength + 1;
      if (stopSeqPos >= static_cast<length_type>(sequence2.size())) {
	break; // avoid switching MAF blocks at the moment
      }
      char c1 = sequence1[startSeqPos];
      char c2 = sequence2[stopSeqPos];
      if (complement) {
	c1 = RnaSecondaryStructureTools::generateComplementDna(c1);
	c2 = RnaSecondaryStructureTools::generateComplementDna(c2);
      }
      // bool isComplement = RnaSecondaryStructureTools::isWatsonCrick(c1, c2, guAllowed);
      bool isComplement = NucleotideTools::isComplementary(c1, c2, guAllowed, false);
      if (!isComplement) {
	if (result == 0) {
	  cout << "Stem " << stem << " containts non-Watson-Crick base pairs." << endl;
	  ASSERT(guAllowed);
	}
	break;
      }
    }
    --result;
    ASSERT(result >= 0);
    // REMARK << "Finished " << methodName << endl;
    ASSERT(result == stemExpandability2Slow(maf,maf2,stem, guAllowed, complement)); // has to match slower and tested version
    return result;
  } 


/** Returns true iff two column pairs are Watson-Crick complementary.
 * Only takes assemblies into account that are present at both columns.
 */
bool
MAFAlignmentTools::isColumnPairComplementary(
     const MAFAlignment& maf,
     length_type colid1,
     length_type colid2,
     bool allowGu,
     bool allowGap) {
  // ERROR_IF(!allowGu, "Should never be called without allowGU!");
  PRECOND(colid1 >= 0 && colid1 < maf.getTotalLength());
  PRECOND(colid2 >= 0 && colid2 < maf.getTotalLength());
  PRECOND(!allowGap);
  length_type aliId1 = maf.getAlignmentId(colid1);
  length_type aliId2 = maf.getAlignmentId(colid2);
  set<string> assemblies = maf.getCommonAssemblies(aliId1, aliId2);
  string slice1 = maf.getSlice(colid1, assemblies);
  string slice2 = maf.getSlice(colid2, assemblies);
  ASSERT(slice1.size() == slice2.size());
  ASSERT(slice1.size() == assemblies.size());
  return NucleotideTools::isComplementary(slice1, slice2, allowGu, allowGap);
}

/** Returns true iff two column pairs are mostly Watson-Crick complementary.
 * Only takes assemblies into account that are present at both columns.
 * @param basePairTypeMin minimum number of allowed base pairs. If set to 1, not checking necessary. If 2 or higher: there must be some 
 *        compensatory base changes.
 */
bool
MAFAlignmentTools::isColumnPairAlmostComplementary(const MAFAlignment& maf,
						   length_type colid1,
						   length_type colid2,
						   double allowedGuFrac,
						   double allowedGapFrac,
                                                   bool convertToComplement,
						   size_type basepairTypeMin) {
  // ERROR_IF(!allowGu, "Should never be called without allowGU!");
  // cout << "Starting isColumnPairAlmostComplementary with " << colid1 << " " << colid2 << " " << allowedGuFrac << " " << allowedGapFrac << " " 
  // << convertToComplement << " " << basepairTypeMin << endl;
  PRECOND(colid1 >= 0 && colid1 < maf.getTotalLength());
  PRECOND(colid2 >= 0 && colid2 < maf.getTotalLength());
  PRECOND(allowedGapFrac >= 0 && allowedGapFrac <= 1.0);
  PRECOND(allowedGuFrac >= 0 && allowedGuFrac <= 1.0);
  // if ((allowedGuFrac == 0.0) && (allowedGapFrac == 0.0)) {
  //   if (!convertToComplement) {
  //     return isColumnPairComplementary(maf,colid1,colid2,false,false);
  //   } else {
  //     return isColumnPairEquivalent(maf,colid1,colid2);
  //   }
  // }
  length_type aliId1 = maf.getAlignmentId(colid1);
  length_type aliId2 = maf.getAlignmentId(colid2);
  set<string> assemblies = maf.getCommonAssemblies(aliId1, aliId2);
  string slice1 = maf.getSlice(colid1, assemblies);
  if (convertToComplement) {
    // cout << "Converting slice " << slice1 << " to complement: " <<  NucleotideTools::dnaComplement(slice1) << endl;
    slice1 = NucleotideTools::dnaComplement(slice1);
  }
  string slice2 = maf.getSlice(colid2, assemblies);
  ASSERT(slice1.size() == slice2.size());
  ASSERT(slice1.size() == assemblies.size());
  bool result = NucleotideTools::isAlmostComplementary(slice1, slice2, allowedGuFrac, allowedGapFrac, basepairTypeMin);
  // if (result) {
  //   cout << "Success in isColumnPairAlmostComplementary with " << colid1 << " " << colid2 << " " << allowedGuFrac << " " << allowedGapFrac 
  // 	 << " " << convertToComplement << " " << basepairTypeMin << " : " << slice1 << " | " << slice2 << endl;
  // }
  return result;
}

/** Returns common assemblies from two different maf blocks
 */
set<string>
MAFAlignmentTools::findCommonAssemblies(
     const MAFAlignment& maf,
     const MAFAlignment& maf2,
     length_type aliId1,
     length_type aliId2) {
  PRECOND(aliId1 < static_cast<length_type>(maf.size()));
  PRECOND(aliId2 < static_cast<length_type>(maf2.size()));
  const map<string, row_type>& map1 = maf.getAssemblyRowIdMap(aliId1);
  const map<string, row_type>& map2 = maf2.getAssemblyRowIdMap(aliId2);
  set<string> result;
  for (map<string,row_type>::const_iterator it = map1.begin(); it != map1.end(); it++) {
    if (map2.find(it->first) != map2.end()) {
      result.insert(it->first);
    }
  }
  return result;
}

/** Returns true iff two column pairs originating from 2 different MAFs
 *  are Watson-Crick complementary.
 * Only takes assemblies into account that are present at both columns.
 */
bool
MAFAlignmentTools::isColumnPairComplementary(const MAFAlignment& maf,
					     const MAFAlignment& maf2,
					     length_type colid1,
					     length_type colid2,
					     bool allowGu,
					     bool allowGap) {
  PRECOND(colid1 >= 0 && colid1 < maf.getTotalLength());
  PRECOND(colid2 >= 0 && colid2 < maf2.getTotalLength());
  length_type aliId1 = maf.getAlignmentId(colid1);
  length_type aliId2 = maf2.getAlignmentId(colid2);
  set<string> assemblies = findCommonAssemblies(maf,maf2,
						aliId1, aliId2);
  string slice1 = maf.getSlice(colid1, assemblies);
  string slice2 = maf2.getSlice(colid2, assemblies);
  ASSERT(slice1.size() == slice2.size());
  ASSERT(slice1.size() == assemblies.size());
  return NucleotideTools::isComplementary(slice1, slice2, allowGu, allowGap);
}

/** Returns true iff two column pairs contain the same nucleotides at non-gap positions.
 * Only takes assemblies into account that are present at both columns.
 */
bool
MAFAlignmentTools::isColumnPairEquivalent(const MAFAlignment& maf, length_type colid1, length_type colid2) {
  PRECOND(colid1 >= 0 && colid1 < maf.getTotalLength());
  PRECOND(colid2 >= 0 && colid2 < maf.getTotalLength());
  length_type aliId1 = maf.getAlignmentId(colid1);
  length_type aliId2 = maf.getAlignmentId(colid2);
  set<string> assemblies = maf.getCommonAssemblies(aliId1, aliId2);
  string slice1 = maf.getSlice(colid1, assemblies);
  string slice2 = maf.getSlice(colid2, assemblies);
  ASSERT(slice1.size() == slice2.size());
  ASSERT(slice1.size() == assemblies.size());
  return slice1 == slice2; // FIXIT: does not work for lower-upper case differences
}

/** Returns true iff two column pairs contain the same nucleotides at non-gap positions.
 * Only takes assemblies into account that are present at both columns.
 */
bool
MAFAlignmentTools::isColumnPairEquivalent(const MAFAlignment& maf, const MAFAlignment& maf2,
                                          length_type colid1, length_type colid2) {
  PRECOND(colid1 >= 0 && colid1 < maf.getTotalLength());
  PRECOND(colid2 >= 0 && colid2 < maf2.getTotalLength());
  length_type aliId1 = maf.getAlignmentId(colid1);
  length_type aliId2 = maf2.getAlignmentId(colid2);
  set<string> assemblies = findCommonAssemblies(maf,maf2,aliId1, aliId2);
  string slice1 = maf.getSlice(colid1, assemblies);
  string slice2 = maf2.getSlice(colid2, assemblies);
  ASSERT(slice1.size() == slice2.size());
  ASSERT(slice1.size() == assemblies.size());
  return slice1 == slice2; // FIXIT: does not work for lower-upper case differences
}


/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
 * If bowth stem strands are from the same MAF, provide it for both maf and maf2
 */
void
MAFAlignmentTools::augmentStemSequence(const MAFAlignment& maf, const MAFAlignment& maf2, Stem& stem, bool reverseMode) {
  string methodName = "augmentStemSequence";
  // REMARK << "Starting " << methodName << endl;
  Stem::index_type newStart = stem.getStart();
  Stem::index_type newStop = stem.getStop();
  ASSERT(newStart > 0);
  ERROR_IF(newStart < maf.getRefAssemblyBegin(),
	   "Internal error in augmentStemSequence: too small start position");
  ERROR_IF(newStart >= maf.getRefAssemblyEnd(),
	   "Internal error in augmentStemSequence: too high start position");
  ERROR_IF(newStop < maf2.getRefAssemblyBegin(),
	   "Internal error in augmentStemSequence: too small stop position");
  ERROR_IF(newStop >= maf2.getRefAssemblyEnd(),
	   "Internal error in augmentStemSequence: too high stop position");

  string seq1, seq2;
  if (reverseMode) { 
    ERROR_IF((newStart+stem.getLength() -1) >= maf.getRefAssemblyEnd(),
	     "Internal error in augmentStemSequence (L60): too high position");
    seq1 = maf.extractAssemblySequence(newStart, newStart + stem.getLength()-1);
    // REMARK << "Extracted first sequence: " << seq1 << endl;
    ERROR_IF((newStop-stem.getLength() +1) < maf2.getRefAssemblyBegin(),
	     "Internal error in augmentStemSequence (L64): too small stop position");
    seq2 = maf2.extractAssemblySequence(newStop - stem.getLength()+1, newStop);
  } else {
    ERROR_IF((newStart+stem.getLength() -1) >= maf.getRefAssemblyEnd(),
	     "Internal error in augmentStemSequence (L68): too large start position");
    seq1 = maf.extractAssemblySequence(newStart, newStart + stem.getLength()-1);
    // REMARK << "Extracted first sequence: " << seq1 << endl;
    ERROR_IF((newStop+stem.getLength() -1) >= maf2.getRefAssemblyEnd(),
	     "Internal error in augmentStemSequence (L72): too large stop position");
    seq2 = maf2.extractAssemblySequence(newStop, newStop +  stem.getLength()-1);
  }
  // REMARK << "Extracted second sequence: " << seq2 << endl;
  ASSERT(seq1.size() == seq2.size());
  ASSERT(static_cast<length_type>(seq1.size()) == stem.getLength());
  stem.setSequence1(seq1);
  stem.setSequence2(seq2);
  // REMARK << "Finished " << methodName << endl;
}

/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
 */
int
MAFAlignmentTools::stemExpandability1Slow(const MAFAlignment& maf, const MAFAlignment& maf2, 
	  const Stem& stem, bool guAllowed, bool complementMode) {
  PRECOND(guAllowed);
  PRECOND(complementMode);
  string methodName = "stemExpandability1Slow";
  // REMARK << "Starting " << methodName << endl;
  int result = 0;
  length_type lowest1 = maf.getRefAssemblyBegin();
  length_type highest1 = maf.getRefAssemblyEnd();
  length_type lowest2 = maf2.getRefAssemblyBegin();
  length_type highest2 = maf2.getRefAssemblyEnd();
  for (result = 0; result < stem.getStart(); ++result) {
    Stem::index_type newStart = stem.getStart() - result;
    Stem::index_type newStop  = stem.getStop()  + result;
    ASSERT(newStart > 0);
    if ((newStart < lowest1) || (newStart >= highest1)) { // undefined otherwise
      break;
    }
    if ((newStop < lowest2) || (newStop >= highest2)) { // undefined otherwise
      break;
    }
    string seq1 = maf.extractAssemblySequence(newStart, newStart);
    string seq2 = maf2.extractAssemblySequence(newStop, newStop);
    if ((seq1.size() != 1) || (seq2.size() != 1)) {
      break;
    }
    if (!complementMode) {
      cout << "Trying to compare " << seq1 << " " << seq2 << endl;
      ERROR("Sorry, currently only reverse-complement search is implemented.");
      //       seq1 = RnaSecondaryStructureTools::generateComplementDna(seq1);
      // seq2 = RnaSecondaryStructureTools::generateComplementDna(seq2);
    }
    ASSERT(seq1.size() == 1 && seq2.size() == 1);
    // bool isComplement = RnaSecondaryStructureTools::isReverseComplement(seq1, seq2, guAllowed);
    bool isComplement = NucleotideTools::isReverseComplementary(seq1, seq2, guAllowed, false);
    if (!isComplement) {
      if (result == 0) {
	cout << "Stem " << stem << " contains non-Watson-Crick base pairs: " 
	     << seq1 << " " << seq2 << endl;
	ASSERT(guAllowed);
      }
      break;
    }
  }
  if (result > 0) {
    --result;
  } else {
    REMARK << "Strange: first base pair of stem is not complementary: " << stem << endl;
    assert(false);
    // assert(guAllowed); 
  }
  ASSERT(result >= 0);
  // REMARK << "Finished " << methodName << endl;
  return result;
} 


/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its aligned sequences are still complementary (no GU).
 */
int
MAFAlignmentTools::stemConservedExpandabilitySlow(
						  const MAFAlignment& maf,
						  const MAFAlignment& maf2, 
						  const Stem& stem,
						  bool leftMode,
						  bool complementMode,
						  bool reverseMode,
						  bool allowGu) {
  string methodName = "stemConservedExpandability1Slow";
  // REMARK << "Starting " << methodName << endl;
  int result = 0;
  length_type lowest1 = maf.getRefAssemblyBegin();
  length_type highest1 = maf.getRefAssemblyEnd();
  length_type lowest2 = maf2.getRefAssemblyBegin();
  length_type highest2 = maf2.getRefAssemblyEnd();
  index_type newStart,newStop;
  for (result = 0; ; ++result) {
    if (reverseMode) {
      if (leftMode) {
	newStart = stem.getStart() - result;
	newStop  = stem.getStop()  + result;
      }
      else {
	index_type newLength = stem.getLength() + result -1;
	newStart = stem.getStart() + newLength;
	newStop = stem.getStop()  - newLength;
      }
    } else { // forward matches
      if (leftMode) {
	newStart = stem.getStart() - result;
	newStop  = stem.getStop()  - result;
      }
      else {
	newStart = stem.getStart() + stem.getLength() -1 + result;
	newStop  = stem.getStop()  + stem.getLength() -1 + result;
      }
    }
    if ((newStart < lowest1) || (newStart >= highest1)) { // undefined otherwise
      break;
    }
    if ((newStop < lowest2) || (newStop >= highest2)) { // undefined otherwise
      break;
    }
    string seq1 = maf.extractAssemblySequence(newStart, newStart);
    string seq2 = maf2.extractAssemblySequence(newStop, newStop);
    if ((seq1.size() != 1) || (seq2.size() != 1)) {
      break;
    }
    ASSERT(seq1.size() == 1 && seq2.size() == 1);
    length_type internalStart = maf.convertAssemblyPositionToColumnId(newStart);
    length_type internalStop = maf2.convertAssemblyPositionToColumnId(newStop);
    if (complementMode) {
      bool isComplement = isColumnPairComplementary(maf, maf2, 
						    internalStart, internalStop, allowGu, false);
      if (!isComplement) {
	if (result == 0) {
	  cout << "Stem " << stem << " contains non-Watson-Crick base pairs." << endl;
	  ASSERT(allowGu);
	}
	break;
      }
    } else {
      bool isMatch = isColumnPairEquivalent(maf, maf2, 
		    internalStart, internalStop);
      if (!isMatch) {
	if (result == 0) {
	  cout << "Forward-Stem " << stem << " is strange: its corresponding sequences are not matching: "
	       << seq1 << " " << seq2 << endl;
	  ASSERT(false);
	}
	break;
      }
    }  
  }
  if (result > 0) {
    --result;
  } else {
    REMARK << "Strange: first base pair of stem is not complementary: " << stem << endl;
    // assert(false);
  }
  ASSERT(result >= 0);
  // REMARK << "Finished " << methodName << endl;
  return result;
} 

/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary.
 */
int
MAFAlignmentTools::stemExpandability2Slow(const MAFAlignment& maf, const MAFAlignment& maf2,
					  const Stem& stem, bool guAllowed, bool complementMode) {
  PRECOND(maf.getRefAssembly() == maf2.getRefAssembly());
  string methodName = "stemExpandability2Slow";
  // REMARK << "Starting " << methodName << endl;
  int result = 0;
  length_type lowest1 = maf.getRefAssemblyBegin();
  length_type highest1 = maf.getRefAssemblyEnd();
  length_type lowest2 = maf2.getRefAssemblyBegin();
  length_type highest2 = maf2.getRefAssemblyEnd();

  for (result = 0; result < stem.getStop(); ++result) {
    Stem::index_type newLength = stem.getLength() + result - 1;
    index_type newStart = stem.getStart() + newLength;
    index_type newStop = stem.getStop()  - newLength;
    if ((newStart < lowest1) || (newStart >= highest1)) { // undefined otherwise
      break;
    }
    if ((newStop < lowest2) || (newStop >= highest2)) { // undefined otherwise
      break;
    }
    string seq1 = maf.extractAssemblySequence(newStart, newStart);
    string seq2 = maf2.extractAssemblySequence(newStop, newStop);
    if ((seq1.size() != 1) || (seq2.size() != 1)) {
      break;
    }
    
    if (!complementMode) {
      cout << "Trying to compare " << seq1 << " " << seq2 << endl;
      ERROR_IF(!complementMode,
	       "Sorry, currently only reverse complement is implemented.");
      //       seq1 = RnaSecondaryStructureTools::generateComplementDna(seq1);
      //       seq2 = RnaSecondaryStructureTools::generateComplementDna(seq2);
    }
    ASSERT(seq1.size() == 1 && seq2.size() == 1);
    // bool isComplement = RnaSecondaryStructureTools::isReverseComplement(seq1, seq2, guAllowed);
    bool isComplement = NucleotideTools::isReverseComplementary(seq1, seq2, guAllowed, false);
    if (!isComplement) {
      if (result == 0) {
	cout << "Stem " << stem << " contains non-Watson-Crick base pairs." << endl;
	ASSERT(guAllowed);
      }
      break;
    }
  }
  if (result > 0) {
    --result;
  } else {
    REMARK << "Strange: last base pair of stem is not complementary: " << stem << endl;
    assert(false);
  }
  ASSERT(result >= 0);
  // REMARK << "Finished " << methodName << endl;
  return result;
} 

/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary. Not regular stem (that would be 
 * method stemExpandabilitySlow1/2), but "forward-matchting stem".
 */
int
MAFAlignmentTools::forwardMatchingStemExpandability1Slow(
			 const MAFAlignment& maf, const MAFAlignment& maf2, 
			 const Stem& stem, bool guAllowed, bool complementMode) {
  string methodName = "stemExpandability1Slow";
  // REMARK << "Starting " << methodName << endl;
  Stem::index_type result = 0;
  length_type lowest1 = maf.getRefAssemblyBegin();
  length_type highest1 = maf.getRefAssemblyEnd();
  length_type lowest2 = maf2.getRefAssemblyBegin();
  length_type highest2 = maf2.getRefAssemblyEnd();

  for (result = 0; (result <= stem.getStart())&&(result <= stem.getStop()); ++result) {
    Stem::index_type newStart = stem.getStart() - result;
    Stem::index_type newStop  = stem.getStop()  - result;
    ASSERT(newStart >= 0);
    ASSERT(newStop >= 0);
    if ((newStart < lowest1) || (newStart >= highest1)) { // undefined otherwise
      break;
    }
    if ((newStop < lowest2) || (newStop >= highest2)) { // undefined otherwise
      break;
    }
    string seq1 = maf.extractAssemblySequence(newStart, newStart);
    string seq2 = maf2.extractAssemblySequence(newStop, newStop);
    if ((seq1.size() != 1) || (seq2.size() != 1)) {
      break;
    }
//     if (complementMode) {
//       seq1 = RnaSecondaryStructureTools::generateComplementDna(seq1);
//       seq2 = RnaSecondaryStructureTools::generateComplementDna(seq2);
//     }
    ASSERT(seq1.size() == 1 && seq2.size() == 1);
    bool isComplement = false;
    if (complementMode) {
      isComplement = NucleotideTools::isComplementary(seq1, seq2, guAllowed, false);
    } else {
      isComplement = (seq1 == seq2); // does not account for GU ! // FIXIT
    }
    if (!isComplement) {
      if (result == 0) {
	cout << "Stem " << stem << " is strange: its corresponding sequences are not forward matches: "
	     << seq1 << " " << seq2 << endl;
      }
      break;
    }
  }
  if (result > 0) {
    --result;
  } else {
    REMARK << "Strange: first base pair of stem is not matching: " << stem << endl;
    assert(false);
  }
  ASSERT(result >= 0);
  // REMARK << "Finished " << methodName << endl;
  return result;
} 


/** For a stem (coordinates given in assembly coordinates, not internal coordinates)
 * determine how far the stem sequences can be expanded in both directions such that its sequences are still complementary. Not regular stem (that would be 
 * method stemExpandabilitySlow1/2), but "forward-matchting stem".
 */
int
MAFAlignmentTools::forwardMatchingStemExpandability2Slow(
		 const MAFAlignment& maf, const MAFAlignment& maf2, 
		 const Stem& stem, bool guAllowed, bool complementMode) {
  string methodName = "stemExpandability2Slow";
  // REMARK << "Starting " << methodName << " with stem " << stem << endl;
  Stem::index_type result = 0;
  length_type lowest1 = maf.getRefAssemblyBegin();
  length_type highest1 = maf.getRefAssemblyEnd();
  length_type lowest2 = maf2.getRefAssemblyBegin();
  length_type highest2 = maf2.getRefAssemblyEnd();

  for (result = 0; result <= maf.getTotalLength(); ++result) {
    Stem::index_type newStart = stem.getStart() + stem.getLength() -1 + result;
    Stem::index_type newStop  = stem.getStop()  + stem.getLength() -1 + result;
    // REMARK << "Trying position pair " << (newStart + 1) <<  " " << (newStop + 1) << endl;
    ASSERT(newStart >= 0);
    ASSERT(newStop >= 0);
    if ((newStart < lowest1) || (newStart >= highest1)) { // undefined otherwise
      break;
    }
    if ((newStop < lowest2) || (newStop >= highest2)) { // undefined otherwise
      break;
    }
    string seq1 = maf.extractAssemblySequence(newStart, newStart);
    string seq2 = maf2.extractAssemblySequence(newStop, newStop);
    // REMARK << "Counter: " << result << " Extracted sequences for positions " << (newStart + 1) << " " 
    // << (newStop + 1) << " : " << seq1 << " " << seq2 << endl;
    if ((seq1.size() != 1) || (seq2.size() != 1)) {
      break;
    }
    bool isComplement = false;
    if (complementMode) {
      // seq1 = RnaSecondaryStructureTools::generateComplementDna(seq1);
      // seq2 = RnaSecondaryStructureTools::generateComplementDna(seq2);
      // REMARK << "Generated complement: " << seq1 << " " << seq2 << endl;
      isComplement = NucleotideTools::isReverseComplementary(seq1,seq2, guAllowed, false);
    } else {
      isComplement = (seq1 == seq2); // does not account for GU; FIXIT
    }
    ASSERT(seq1.size() == 1 && seq2.size() == 1);

    if (!isComplement) {
      if (result == 0) {
	cout << "Stem " << stem 
	     << " is strange: its corresponding sequences are not forward matches: "
	     << seq1 << " " << seq2 << endl;
      }
      break;
    }
  }
  if (result > 0) {
    --result;
  } else {
    REMARK << "Strange: last base pair of stem is not matching: " << stem << endl;
    assert(false);
  }
  ASSERT(result >= 0);
  // REMARK << "Finished " << methodName << " with result: " << result << endl;
  return result;
} 
