#ifndef _RNASECONDARY_STRUCTURE_TOOLS_H_
#define _RNASECONDARY_STRUCTURE_TOOLS_H_

#include <iostream>
#include <string>
#include <Vec.h>
#include <Stem.h>
#include <stemhelp.h>
#include <StringTools.h>
#include <vectornumerics.h>
#include <SimpleSequenceAlignment.h>

class RnaSecondaryStructureTools {

 public:

  typedef Stem stem_type;
  typedef Vec<Stem> stem_set;
  typedef RnaSecondaryStructure::index_type index_type;
  typedef Vec<index_type> mapping_container;
  typedef RnaSecondaryStructure::size_type size_type;
  typedef RnaSecondaryStructure::sequence_type sequence_type;

  /** Returns true, iff all residues involved in the stem are so far not participating in a base pair */
  static bool isUnpaired(const RnaSecondaryStructure& structure, const Stem& stem) {
    index_type start = stem.getStart();
    index_type stop = stem.getStop();
    index_type seqLen = static_cast<index_type>(structure.getSequence().size());
    for (index_type i = 0; i < stem.getLength(); ++i) {
      ASSERT(start + i < seqLen);
      ASSERT(stop - i >= 0);
      if (structure.isBasePaired(start + i) || structure.isBasePaired(stop-i)) {
	return false;
      }
    }
    return true;
  }

  /** Generates RNASecondaryStructure from input file in CT format (single structure) */
  static RnaSecondaryStructure readCt(istream& is) {
    Vec<Stem> stems;
    string sequence;
    readCtSingle(is, stems, sequence, 1); // function from stemhelp
    return RnaSecondaryStructure(stems, sequence);
  }

  /** Reards and parses RNAfold output file */
  static RnaSecondaryStructure readFastaSecondary(istream& is) {
    SimpleSequenceAlignment sali;
    sali.readFasta(is);
    string sequence;
    string bracket;
    Vec<index_type> starts(1, 0);
    for (SequenceAlignment::size_type i = 0; i < sali.size(); ++i) {
      string s = sali.getSequence(i);
      string s1 = s.substr(0, s.size()/2);
      string s2 = s.substr(s.size()/2, s.size()/2);
      ASSERT(s1.size() + s2.size() == s.size());
      sequence = sequence + s1;
      bracket = bracket + s2;
      starts.push_back(starts[starts.size()-1] + s1.size());
    }
    cout << sequence << endl << bracket << endl << starts << endl;
    unsigned int pkCounter = 0;
    Vec<Vec<double> > matrix = secToMatrix2(bracket, 1.0, pkCounter);
    cout << "Debug: matrix count of ones: "<< matrixCount(matrix, 1.0) << endl;
    Vec<Stem> stems = generateStemsFromMatrix(matrix, 1, 0.5, sequence);
    return RnaSecondaryStructure(stems, sequence, starts);
  }

  /** Reads and parses RNAfold output file */
  static RnaSecondaryStructure readFastaSecondaryCosts(istream& is) {
    SimpleSequenceAlignment sali;
    sali.readFasta(is);
    string sequence;
    string bracket;
    string costsString;
    double costExponent = 3.0;
    Vec<index_type> starts(1, 0);
    for (SequenceAlignment::size_type i = 0; i < sali.size(); ++i) {
      string s = sali.getSequence(i);
      string s1 = s.substr(0, s.size()/3);
      string s2 = s.substr(s.size()/3, s.size()/3);
      string s3 = s.substr(2 * (s.size()/3), s.size()/3);
      ASSERT(s1.size() + s2.size() + s3.size() == s.size());
      sequence = sequence + s1;
      bracket = bracket + s2;
      costsString = costsString + s3;
      starts.push_back(starts[starts.size()-1] + s1.size());
    }
    Vec<double> costs(costsString.size(), 0.0);
    for (string::size_type i = 0; i < costsString.size(); ++i) {
      costs[i] = (10.0/9.0) * stod(costsString.substr(i,1)); // number between 0 and 9 to 0.0 .. 1.0
      if (costs[i] > 1.0) {
	costs[i] = 1.0;
      }
      costs[i] = pow(costs[i], costExponent);
    }
    cout << "Debug sequence, bracket and starts: " << sequence << endl << bracket << endl << starts << endl;
    unsigned int pkCounter = 0;
    Vec<Vec<double> > matrix = secToMatrix2(bracket, 1.0, pkCounter);
    cout << "Debug: matrix count of ones: "<< matrixCount(matrix, 1.0) << endl;
    Vec<Stem> stems = generateStemsFromMatrix(matrix, 1, 0.5, sequence);
    cout << "Debug: Generated stems from bracket notation: " << stems << endl;
    RnaSecondaryStructure result(stems, sequence, starts);
    result.setDesignCosts(costs);
    return result;
  }

  /** Reards and parses RNAfold output file */
  static RnaSecondaryStructure readRNAfold(istream& is) {
    string line = getLine(is);
    ERROR_IF(line.size() == 0, "readRNAfoldUnexcected empty first line encountered.");
    if (line[0] == '>') {
      line = getLine(is); // skip header line
    }
    string sequence = line;
    ERROR_IF(sequence.size() == 0, "Sequence expected in first line after optional header in RNAfold format!");
    string bracket = getLine(is);
    unsigned int pkCounter = 0;
    Vec<Vec<double> > matrix = secToMatrix(bracket, 1.0, pkCounter);
    Vec<Stem> stems = generateStemsFromMatrix(matrix, 1, 0.5, sequence);
    return RnaSecondaryStructure(stems, sequence);
  }


  /** returns true if AU, GC, GU (or T instead of U) 
   * TODO: not hard-coded characters!
   */
  static bool isWatsonCrick(char c1, char c2, bool guAllowed=false) {
    c1 = toupper(c1);
    if (c1 == 'T') {
      c1 = 'U';
    }
    c2 = toupper(c2);
    if (c2 == 'T') {
      c2 = 'U';
    }
    switch (c1) {
    case 'A':
      return (c2 == 'U');
    case 'C':
      return (c2 == 'G');
    case 'G':
      return ((c2 == 'C') || (guAllowed && (c2 == 'U')));
    case 'U':
      return ((c2 == 'A') || (guAllowed && (c2 == 'G')));
    default: return false;
    }
    return false;
  }

  /** Test isWatsonCrick method */
  static bool isWatsonCrickTest() {
    return isWatsonCrick('A','T', false) && isWatsonCrick('T','A', false) && isWatsonCrick('C','G', false) && isWatsonCrick('G','C', false) && (!isWatsonCrick('G','T', false)) && (!isWatsonCrick('T','G', false))
      && isWatsonCrick('G','T', true) && isWatsonCrick('T','G', true) && (!isWatsonCrick('A','C', false))
      && (!isWatsonCrick('A','G', false)) && (!isWatsonCrick('A','A', false));
  }

  /** Returns RNA base pair complementary character
   */
  static char generateComplementRna(char c1) {
    c1 = toupper(c1);
    switch (c1) {
    case 'A':
      return ('U');
    case 'C':
      return ('G');
    case 'G':
      return ('C');
    case 'U':
      return ('A');
    case 'T':
      return ('A');
    }
    return 'X';
  }
  
  /** Returns RNA base pair complementary character
   */
  static char generateComplementDna(char c1) {
    c1 = toupper(c1);
    switch (c1) {
    case 'A':
      return ('T');
    case 'C':
      return ('G');
    case 'G':
      return ('C');
    case 'T':
      return ('A');
    case 'U':
      return ('A');
    }
    return 'X';
  }

  /** Returns complementary DNA
   */
  static string generateComplementDna(const string& s) {
    string result = s;
    for (string::size_type i = 0; i < result.size(); ++i) {
      result[i] = generateComplementDna(s[i]);
      INVARIANT(isWatsonCrick(result[i], s[i], false));
    }
    return result;
  }

  /** Returns true if test for generating complementary DNA is passed
   */
  static bool generateComplementDnaTest() {
    return generateComplementDna("ACAGT") == "TGTCA";
  }

  /** Returns RNA base pair complementary character
   */
  static string generateComplementRna(const string& s) {
    string result = s;
    for (string::size_type i = 0; i < result.size(); ++i) {
      result[i] = generateComplementRna(s[i]);
      INVARIANT(isWatsonCrick(result[i], s[i], false));
    }
    return result;
  }


  /** returns true if AU, GC, GU (or T instead of U) for whole strings
   * TODO: not hard-coded characters!
   */
  static bool isReverseComplement(const string& s1, const string& s2, bool guAllowed=false) {
    ASSERT(s1.size() == s2.size());
    string::size_type n = s1.size(); 
    for (string::size_type i = 0; i < s1.size(); ++i) {
      if (!isWatsonCrick(s1[i], s2[n - i - 1], guAllowed)) {
	return false;
      }
    }
    return true;
  }

  /** returns true if AU, GC, GU (or T instead of U) for whole strings
   * TODO: not hard-coded characters!
   */
  static bool isReverseComplementTest() {
    return isReverseComplement("AGCTTC", "GAAGCT");
  }

  /** Maps a secondary structure between index sets. Useful for mapping a secondary structure
   * betwenn gapped and ungapped sequence.
   * mapping[i] = i2 : i : index in old structure, i2: index in new structure. */
  template <typename T>
  static RnaSecondaryStructure mapStructure(const RnaSecondaryStructure& other,
					    const string& newSeq, const T& mapping) {
    ASSERT(mapping.size() == other.size()); // mapping[i] = i2
    ASSERT(mapping.size() > 0);
    RnaSecondaryStructure newStruct(newSeq);
    for (index_type i = 0; i < other.getLength(); ++i) {
      if (other.isPaired(i)) {
	index_type j = other.getBasePair(i);
	ASSERT((j >= 0) && (j < other.getLength()));
	if(!isWatsonCrick(newSeq[mapping[i]], newSeq[mapping[j]])) {
	  cout << "Weird mapping:" << endl;
	  cout << other.getSequence() << endl;
	  cout << newSeq << endl << mapping << endl;
	  cout << "mapping: " << generateMappingFromUngappedToGapped(newSeq) << endl; // other.getSequence()) << endl;
	}
	ASSERT(isWatsonCrick(newSeq[mapping[i]], newSeq[mapping[j]]));
	newStruct.setPaired(mapping[i], mapping[j]);
      }
    }
    return newStruct;
  }
  
  static mapping_container generateMappingFromGappedToUngapped(const sequence_type& gappedSeq) {
    mapping_container result(gappedSeq.size(), RnaSecondaryStructure::UNPAIRED_ID);
    index_type pc = 0;
    for (size_type i = 0; i < gappedSeq.size(); ++i) {
      if ((gappedSeq[i] != '.') && (gappedSeq[i] != '-')) {
	result[i] = pc++;
      }
    }
    ASSERT(result.size() == gappedSeq.size());
    return result;
  }

  static mapping_container generateMappingFromUngappedToGapped(const sequence_type& gappedSeq) {
    // cout << "Gapped: " << gappedSeq << endl;
    sequence_type ungapped = removeFromString(gappedSeq, string("-."));
    // cout << "Gapped, ungapped (2): " << gappedSeq << endl << ungapped << endl;
    mapping_container result(ungapped.size(), RnaSecondaryStructure::UNPAIRED_ID);
    size_type pc = 0;
    for (size_type i = 0; i < gappedSeq.size(); ++i) {
      if ((gappedSeq[i] != '.') && (gappedSeq[i] != '-')) {
	ASSERT(pc < result.size());
	result[pc++] = i;
      }
    }
    ASSERT(result.size() == ungapped.size());
    return result;
  }

  /** returns true, if sequence of RnaSecondaryStructure has a consensus base pair at m,n */
  template<typename T>
  static bool isConsensusPaired(T first, T last, index_type m, index_type n) {
    size_type pairedCount = 0;
    size_type totCount = 0;
    for (T it = first; it != last; it++) {
      if (it->isPaired(m,n)) {
	++pairedCount;
      }
      ++totCount;
    }
    double frac = static_cast<double>(pairedCount) / static_cast<double>(totCount);
    return frac >= 0.5;
  }

  /** returns true, if sequence of RnaSecondaryStructure has a consensus base pair at m,n */
  template<typename T>
  static bool isConsensusPairedFromSecond(T first, T last, index_type m, index_type n) {
    ASSERT(first != last);
    size_type pairedCount = 0;
    size_type totCount = 0;
    for (T it = first; it != last; it++) {
      if (it->second.isPaired(m,n)) {
	++pairedCount;
      }
      ++totCount;
    }
    double frac = static_cast<double>(pairedCount) / static_cast<double>(totCount);
    return frac >= 0.5;
  }

  template<typename T>
  static RnaSecondaryStructure generateConsensusStructure(T first, T last) {
    ASSERT(first != last);
    RnaSecondaryStructure result(first->size());
    result.setSequence(first->getSequence());// FIXIT generateConsensusSequence(first, last));
    for (index_type i = 0; i < result.getLength(); ++i) {
      for (index_type j = i + 1; j < result.getLength(); ++j) {
	if (isConsensusPaired(first, last, i, j)) {
	  result.setPaired(i,j);
	  break;
	} 
      }
    }
    return result;
  }

  template<typename T>
  static RnaSecondaryStructure generateConsensusStructureFromSecond(T first, T last) {
    ASSERT(first != last);
    RnaSecondaryStructure result(first->second.size());
    result.setSequence(first->second.getSequence());// FIXIT generateConsensusSequence(first, last));
    for (index_type i = 0; i < result.getLength(); ++i) {
      for (index_type j = i + 1; j < result.getLength(); ++j) {
	if (isConsensusPairedFromSecond(first, last, i, j)) {
	  result.setPaired(i,j);
	  break;
	} 
      }
    }
    return result;
  }
  
  /** writes RNA secondary structure a Ct format file (same format the is producted by mfold) */
  static void writeCt(ostream& os, 
		      const RnaSecondaryStructure& structure) {
    Vec<Stem> stems = convertSecondaryStructureToStems(structure);
    string sequence = structure.getSequence();;
    writeCtSingle(os, stems, sequence);
  }

  /** generates bracket notation for structure */
  static string toBracket(const RnaSecondaryStructure& structure) {
    Vec<Stem> stems = convertSecondaryStructureToStems(structure);
    return stemsToBracketFasta(stems, structure.size());
  }

  static double computeMatthews(const AbstractSecondaryStructure& query, const AbstractSecondaryStructure& reference) {
    ASSERT(query.size() == reference.size());
    RnaSecondaryStructure::basepair_set_type queryBps = query.getBasePairs();
    RnaSecondaryStructure::basepair_set_type refBps = reference.getBasePairs();
    size_type n = queryBps.size();
    unsigned int truePos = 0;
    unsigned int falsePos = 0;
    unsigned int trueNeg = (n * (n-1))/2;
    unsigned int falseNeg = 0;
    ASSERT(queryBps.size() == refBps.size());
    for (size_type i = 0; i < queryBps.size(); ++i) {
      if (queryBps[i] >= 0) {
	if (refBps[i] >= 0) {
	  if (queryBps[i] == refBps[i]) {
	    ++truePos;
	    --trueNeg;
	  } else {
	    ++falsePos;
	    ++falseNeg;
	    --trueNeg;
	    --trueNeg;
	  }
	} else {
	  ++falsePos;
	  --trueNeg;
	}
      } else if (refBps[i] >= 0) { // reference has a this position a base pair, query not: false negative
	++falseNeg;
	--trueNeg;
      } // no need for trueNeg: already accounted 
    }
    return computeMathews(truePos, falsePos, trueNeg, falseNeg);
  }

};

#endif
