# takes two strings, counts how many characters are complementary (1-1 forward match)
count.complementary <- function(col1, col2, gu.allowed=TRUE) {
  stopifnot(!is.null(col1))
  stopifnot(!is.null(col2))
#  stopifnot(length(col1) == 1)
#  stopifnot(length(col2) == 1)
  n <- nchar(col1)
  stopifnot(nchar(col2) == n)
  stopifnot(n > 0)
  counter <- 0
  for (i in 1:n) {
   if (is.complementary(substr(col1,i,i), substr(col2,i,i), gu.allowed=gu.allowed)) {
    counter <- counter + 1
   }
  }
  counter
}

test.count.complementary <- function(col1="AAGU", col2="UUCU", gu.allowed=TRUE) {
  result <- count.complementary(col1, col2, gu.allowed=gu.allowed)
  cat(col1, col2, result, "\n")
  stopifnot(result == 3)
}

findNonGapRows <- function(col1, col2) {
  n <- nchar(col1)
  stopifnot(n > 0)
  stopifnot(nchar(col2) == n)
  stopifnot(length(col1) == 1)
  stopifnot(length(col2) == 1)
  result <- rep(TRUE, n)
  for (i in 1:n) {
    if (is.gap(substr(col1, i, i)) || is.gap(substr(col2, i,i))) {
      result[i] <- FALSE
    }
  }
  result
}

findAlphabetRows <- function(col1, col2, alphabet) {
  n <- nchar(col1)
  stopifnot(n > 0)
  stopifnot(nchar(col2) == n)
  stopifnot(length(col1) == 1)
  stopifnot(length(col2) == 1)
  result <- rep(TRUE, n)
  chars1 <- strsplit(col1, "")[[1]]
  chars2 <- strsplit(col2, "")[[1]]
  for (i in 1:n) {
    result[i] <- ((!is.na(alphabet[chars1[i]])) && (!is.na(alphabet[chars2[i]])))
  }
  result
}

test.findNonGapRows <- function(col1="AC-G-", col2="A.-GC") {
 result <- findNonGapRows(col1, col2)
 cat("Input columns:",col1, col2, "\n")
 print(result)
 result
}

test.findAlphabetRows <- function(col1="AC-G-A", col2="A.-GCN", alphabet=RNAAlphabet()) {
 result <- findAlphabetRows(col1, col2, alphabet)
 cat("Input columns:",col1, col2, "\n")
 print(result)
 checkTrue(result[1])
 checkTrue(!result[length(result)])
 result
}

stringIdSubstring <- function(s, ids) {
  chars <- strsplit(s, "")[[1]]
  paste(chars[ids], collapse="")
}

test.stringIdSubstring <- function(s="hello", ids=c(FALSE, TRUE, FALSE, FALSE, TRUE)) {
 result <- stringIdSubstring(s, ids)
 print(result)
 stopifnot(result == "eo")
 result
}

# s is a string representing a single sequence
is.conservedSequence <- function(s, gapAllowed=TRUE) {
  s <- toupper(s)
  chars <- strsplit(s, "")[[1]]
  result <- TRUE
  if (gapAllowed) {
    goodVec <- rep(TRUE, length(chars))
    for (i in 1:length(chars)) {
     if (is.gap(chars[i])) {
      goodVec[i] = FALSE
     }
    }
    chars <- chars[goodVec]
  } else {
    for (i in 1:length(chars)) {
     if (is.gap(chars[i])) {
      result <- FALSE
      break
     }
    }
  }
  if (result) {
   if (length(chars) < 2) {
     result <- FALSE
   } else {
     chars <- unique(chars)
     for (i in 1:length(chars)) {
       stopifnot(!is.gap(chars[i]))
     }
     if (length(chars) > 1) {
       result <- FALSE
     }
   }

  }
  result
}

test.is.conservedSequence <- function() {
  checkTrue(is.conservedSequence("AAAA"))
  checkTrue(is.conservedSequence("AAaAa"))
  checkTrue(is.conservedSequence("AAa-Aa"))
  checkTrue(is.conservedSequence("A--A"))
  checkTrue(!is.conservedSequence("A--A", gapAllowed=FALSE))
  checkTrue(!is.conservedSequence("A--"))
  checkTrue(!is.conservedSequence("AAaAaAG"))
  checkTrue(!is.conservedSequence("AAaAaAG"))
  checkTrue(is.conservedSequence("TTTttTT"))
}

count.complementary.bias <- function(col1, col2, iterations=1000, gu.allowed=TRUE, keep=FALSE, pLim=0.1, iterCheck=100, alphabet=RNAAlphabet()) {
  stop("Avoid using count.complementary.bias: rather use complementarity.binom.test")
  nonGapRows <- findAlphabetRows(col1, col2)
  col1 <- toupper(stringIdSubstring(col1, nonGapRows)) # only consider rows that are not gapped
  col2 <- toupper(stringIdSubstring(col2, nonGapRows))
  n0 <- count.complementary(col1, col2, gu.allowed=gu.allowed)
  successes <- 0
  nrv <- rep(NA, iterations)
  effIterations = iterations
  conservedFlag <- FALSE
  if (is.conservedSequence(col1) || is.conservedSequence(col2)) {
    effIterations = 1
    successes = 1
    conservedFlag <- TRUE
  } else {
   for (i in 1:iterations) {
     col1r <- shuffleAlignmentColumnGapPreserved(col1, collapse=TRUE)
     col2r <- shuffleAlignmentColumnGapPreserved(col2, collapse=TRUE)
     stopifnot(nchar(col1) == nchar(col1r))
     stopifnot(nchar(col2) == nchar(col2r))
     nr <- count.complementary(col1r, col2r,gu.allowed=gu.allowed)
     if (keep) {
       nrv[i] <- nr
     }
     if (nr >= n0) {
       successes <- successes + 1
     }
     p <- successes / i
     if (p > pLim) {
      effIterations = i
      break
     }
   }
  }
  pval <- successes / effIterations
  if (successes == 0) {
    pval <- 1/iterations
  }
  result <- list()
  result[["p.value"]] <- pval
  result[["iterations"]] <- effIterations
  result[["successes"]] <- successes
  result[["string1"]] <- col1
  result[["string2"]] <- col2
  result[["conserved"]] <- conservedFlag
  result[["number_characters"]] <- nchar(col1)
  result[["unshuffled_count"]] <- n0
  if (keep) {
   result[["shuffled_counts"]] <- nrv
  }
  result
}

ExtractFreqeuncies <- function(s) {
 lets <- strsplit(s, "")[[1]]
 result <- table(lets) / nchar(s)
 result
}

# counts number of occurrence of each letter combination
ExtractPairFrequencies <- function(s1,s2, alphabet) {
 nl <- length(alphabet)
 result <- matrix(0, ncol=nl, nrow=nl)
 colnames(result) <- alphabet
 rownames(result) <- alphabet
# print(result)
 lets1 <- strsplit(s1, "")[[1]]
 lets2 <- strsplit(s2, "")[[1]]
 n <- length(lets1)
 stopifnot(length(lets2) == n)
 for (i in 1:n) {
  result[lets1[i], lets2[i]] <- result[lets1[i],lets2[i]] + 1
 }
 result
}

GenerateExpectedFrequencies <- function(mtx, alphabet) {
 nr <- nrow(mtx)
 nc <- nrow(mtx)
 nl <- length(alphabet)
 result <- matrix(0, ncol=nc, nrow=nr)
 colnames(result) <- colnames(mtx)
 rownames(result) <- rownames(mtx)
 nToT = sum(mtx)
 for (i in 1:nr) {
  relFreqI <- sum(mtx[i,])/nToT
  for (j in 1:nc) {
   relFreqJ <- sum(mtx[,j])/nToT
   probIJ <- relFreqI * relFreqJ
   expIJ <- probIJ * nToT # this is expected
   result[i,j] <- expIJ
  }
 }
 result
}

ExtractMatrixElements <- function(mtx, pairs) {
 result <- rep(NA, length(pairs))
 for (i in 1:length(pairs)) {
  result[i] <- mtx[pairs[[i]][1], pairs[[i]][2]]
 }
 result
}

BasepairingPairs <- function(TorU, gu.allowed) {
 result <- list( c("A",TorU), 
                 c("C","G"), 
                 c("G", "C"), 
                 c(TorU, "A")) 
 if (gu.allowed) {
  result <- append(result, list(c("G",TorU), c(TorU, "G")))
 }
 result
}

RNAAlphabet <- function() {
 result <- c("A","C","G","U")
 names(result) <- result
 result
}

NucleotideAlphabet <- function(TorU) {
 result <- c("A","C","G",TorU)
 names(result) <- result
 result
}

test.ExtractPairFrequencies <- function(s1="AACG", s2="UUGC", alphabet=c("A","C","G","U")) {
  cat("Creating pair count table for strings", s1, s2, "\n")
  ExtractPairFrequencies(s1, s2, alphabet=alphabet)
}

complementarity.binom.test <- function(col1, col2, gu.allowed=TRUE, alternative="greater") {
#  cat("Starting complementarity.binom.test...\n")
#  print(col1)
#  print(col2)
  if (length(col1) > 1) {
   col1 <- paste(col1, collapse="")
  } 
  if (length(col2) > 1) {
   col2 <- paste(col2, collapse="")
  }
  alphabet=RNAAlphabet()
  n <- nchar(col1)
  # translate T to U:
  col1 <- gsub("T", "U", toupper(col1))
  col2 <- gsub("T", "U", toupper(col2))
  alphabetRows <- findAlphabetRows(col1, col2, alphabet=RNAAlphabet())
  col1 <- stringIdSubstring(col1, alphabetRows) # only consider rows that are not gapped
  col2 <- stringIdSubstring(col2, alphabetRows)
#  cat("Extracting pair frequences for column pair:", col1, ":", col2, "\n")
  result <- list()
  result[["p.value"]] <- 1
  if ((length(col1) == 1) && (nchar(col1) > 0)) {
   mtx <- ExtractPairFrequencies(col1, col2, alphabet)
   mtxe <- GenerateExpectedFrequencies(mtx, alphabet)
   mtxer <- mtxe/sum(mtxe) # probabilities
   expPairingProb = sum(ExtractMatrixElements(mtxer, BasepairingPairs("U", gu.allowed=gu.allowed)))
   foundPairing = sum(ExtractMatrixElements(mtx, BasepairingPairs("U", gu.allowed=gu.allowed)))
   result <- binom.test(foundPairing, nchar(col1), expPairingProb, alternative=alternative)
  } 
  result
}

# generates a "vote", if there is evidence that the compositional bias in a column-pair is
# due to an RNA or DNA interaction. The result is a string, that can have one of 3 values:
# "inconclusive", "RNA", "DNA"
# 
DNAorRNA.vote <- function(col1, col2, pLim=0.05) {
#  cat("Starting complementarity.binom.test...\n")
#  print(col1)
#  print(col2)
  if (length(col1) > 1) {
   col1 <- paste(col1, collapse="")
  } 
  if (length(col2) > 1) {
   col2 <- paste(col2, collapse="")
  }
  alphabet=RNAAlphabet()
  n <- nchar(col1)
  # translate T to U:
  col1 <- gsub("T", "U", toupper(col1))
  col2 <- gsub("T", "U", toupper(col2))
  alphabetRows <- findAlphabetRows(col1, col2, alphabet=RNAAlphabet())
  col1 <- stringIdSubstring(col1, alphabetRows) # only consider rows that are not gapped
  col2 <- stringIdSubstring(col2, alphabetRows)
#  cat("Extracting pair frequences for column pair:", col1, ":", col2, "\n")
  result <- "inconclusive" 
  guBasepairList <- list(c("G","U"), c("U","G"))
  if ((length(col1) == 1) && (nchar(col1) > 0)) {
   mtx <- ExtractPairFrequencies(col1, col2, alphabet)
   mtxe <- GenerateExpectedFrequencies(mtx, alphabet)
   mtxer <- mtxe/sum(mtxe) # probabilities
   expPairingProb = sum(ExtractMatrixElements(mtxer, guBasepairList))
   foundPairing = sum(ExtractMatrixElements(mtx, guBasepairList))
   # if DNA interaction, GU base pairs should be under-represented:
   dnaResult <- binom.test(foundPairing, nchar(col1), expPairingProb, alternative="less")
   # if RNA interaction, GU base pairs should be over-represented:
   rnaResult <- binom.test(foundPairing, nchar(col1), expPairingProb, alternative="greater")
   if ((rnaResult[["p.value"]] <= pLim) && (dnaResult[["p.value"]] > pLim)) {
    result <- "RNA" 
   } else if ((dnaResult[["p.value"]] <= pLim) && (rnaResult[["p.value"]] > pLim)) {
    result <- "DNA" 
   }
   attr(result, "rnaResult") <- rnaResult
   attr(result, "dnaResult") <- dnaResult
  } 
  result
}

# this is a test of a column pair, that appears to avoid GU base pairing
# it should result in a vote for "DNA" interactions
test.DNAOrRNA.vote.1 <- function(col1, col2, pLim=0.1) {
 col1 <- "AAACCCGGGUUUAGCGAU"
 col2 <- "UUUGGGCCCAAAUCGCUA"
 result1 <- DNAorRNA.vote(col1, col2, pLim=pLim)   
 print(result1)
}

# this is a test of a column pair, that appears to favor GU base pairing
# it should result in a vote for "RNA" interactions
test.DNAOrRNA.vote.2 <- function(col1, col2, pLim=0.1) {
 col1 <- "GGGCCCGGGUUUGAU"
 col2 <- "UUUGGGCCCGGGCUG"
 result2 <- DNAorRNA.vote(col1, col2, pLim=pLim)   
 print(result2)
}

# for a matrix if possible covarying alignment columns, compile votes for evidence of RNA or DNA interaction
DNAorRNA.votes.matrix <- function(mtx, charArray1, charArray2, pLimCol=0.05, pLimCons=0.05, selfMode=FALSE) {
	if (is.list(charArray1)) {
		charArray1 <- seqsToCharArray(charArray1, transform.upper=TRUE)
	}
	if (is.list(charArray2)) {
		charArray2 <- seqsToCharArray(charArray2, transform.upper=TRUE)
	}
	nCols <- ncol(mtx)
        nRows <- nrow(mtx)
	stopifnot(nRows == ncol(charArray1))
	stopifnot(nCols == ncol(charArray2))
	dnaVotes <- 0
	rnaVotes <- 0
	incVotes <- 0
        stopifnot(nRows > 0)
        stopifnot(nCols > 0)
	for (i in 1:nRows) {
		startJ = 1
		if (selfMode) {
			startJ <- i + 1
			if (startJ > nCols) {
				break
			}
		}
		col1 <- charArray1[,i]
		for (j in startJ:nCols) {
			if (mtx[i,j] <= pLimCol) {
				# col2 <- charArray1[,j]
				col2 <- charArray2[,j]
				colResult <- DNAorRNA.vote(col1, col2, pLim=pLimCol)
				if (colResult == "RNA") {
					rnaVotes <- rnaVotes + 1
				} else if (colResult == "DNA") {
					dnaVotes <- dnaVotes + 1
				} else if (colResult == "inconclusive") {
					incVotes <- incVotes + 1
				} else {
					stop(paste("Voting result should be RNA, DNA or inconclusive:", colResult))
				}
			}
		}
	}
        totVotes <- dnaVotes + rnaVotes + incVotes
        totRelevantVotes <- dnaVotes + rnaVotes
        finalResult <- list()
        finalResult[["consensus"]] <- "inconclusive"
	finalResult[["RNA_Votes"]] <- rnaVotes
	finalResult[["DNA_Votes"]] <- dnaVotes
	finalResult[["inconclusive_Votes"]] <- incVotes
	if (totRelevantVotes > 0) {
		rnaFrac = rnaVotes / totRelevantVotes
		binomResult <- binom.test(rnaVotes, totRelevantVotes, p=0.5)
		finalResult[["binom"]] <- binomResult
		if (binomResult[["p.value"]] <= pLimCons) {
			if (rnaFrac < 0.5) {
				finalResult[["consensus"]] <- "DNA"
			} else {
				finalResult[["consensus"]] <- "RNA"
			}
		}
	}
#	cat("Result of RNA or DNA votes:\n")
#        print(finalResult)
	finalResult
}

complementarity.binom.test.matrix <- function(colVec1, colVec2=NULL, gu.allowed=TRUE, alternative="greater", combineMethod=NULL, verbose=FALSE) {
  selfMode <- (is.null(colVec2))
  if (is.null(colVec2)) {
    colVec2 = colVec1
  }
  if (is.list(colVec1)) {
   colVec1 <- seqsToCharArray(colVec1, transform.upper=TRUE)
  } 
  if (is.list(colVec2)) {
   colVec2 <- seqsToCharArray(colVec2, transform.upper=TRUE)
  } 
  n1 <- ncol(colVec1)
  n2 <- ncol(colVec2)
  if (verbose) {
   cat("Starting to compute matrix of P-values with dimensions", n1, n2, "\n")
   cat("Self mode: ", selfMode, "\n")
  }
  stopifnot(n1 >= 2)
  result <- matrix(1, nrow=n1, ncol=n2)
  pVec <- c()
  for (i in 1:(n1-1)) {
   start2 <- 1
   if (selfMode) {
    start2 <- i+1
   }
   stopifnot(start2 <= n2)
   for (j in start2:n2) {
       result[i,j] <- complementarity.binom.test(colVec1[,i], colVec2[,j], gu.allowed=gu.allowed, alternative=alternative)[["p.value"]] 
       if (verbose) {
         cat("Set position", i,j, "to", result[i,j], "\n")
       }
       if (selfMode) {
         result[j,i] <- result[i,j]
       }
       pVec[length(pVec) + 1] <- result[i,j]
   }
  }
  if (length(combineMethod) > 0) {
   attr(result, "p.combined") <- computeCombinedPValues(pVec, method=combineMethod) 
  }  
  result
}

test.complementarity.binom.test.matrix <- function(file=system.file("data/tRNA_ecoli_anticodon.fa", package="rnafolding")) {
  seqs <- readFASTA(file)
  result <- complementarity.binom.test.matrix(seqs)
  checkTrue(is.matrix(result))
  checkTrue(nrow(result) == nchar(seqs[[1]]$seq))
  checkTrue(ncol(result) == nchar(seqs[[1]]$seq))
  result
}

test.complementarity.binom.test <- function(s1="AGGU", s2="UCCA") {
 checkTrue(complementarity.binom.test("AAA","CCC")[["p.value"]] == 1)
 checkTrue(complementarity.binom.test("AAA","AAA")[["p.value"]] == 1)
 checkTrue(complementarity.binom.test("UUU","AAA")[["p.value"]] == 1)
 checkTrue(complementarity.binom.test("UUA","AAU")[["p.value"]] < 1)
 checkTrue(complementarity.binom.test("UUG","GGU")[["p.value"]] < 1)
 checkTrue(complementarity.binom.test("UUG","GGU", gu.allowed=FALSE)[["p.value"]] == 1)
 checkTrue(complementarity.binom.test("UUGAC","GGUUG")[["p.value"]] < 0.05)
 checkTrue(complementarity.binom.test("UUGACN","GGUUGA")[["p.value"]] < 0.05)
 checkTrue(complementarity.binom.test("---","NNN")[["p.value"]] == 1)

 checkTrue(complementarity.binom.test("UUGAC","GGUUG")[["p.value"]] == complementarity.binom.test("TTGAC","GGTTG")[["p.value"]])
 checkTrue(complementarity.binom.test("UUGAC","GGUUG")[["p.value"]] == complementarity.binom.test("TTGAC-","GGTTGA")[["p.value"]])
}

# computes highest z-score of found complementarity
count.complementary.highestZ <- function(col1, colVec, ids=NULL, gu.allowed=TRUE, transform.z=TRUE) {
  stopifnot(length(colVec) > 1)
  stopifnot(nchar(col1) > 0)
  col1 <- toupper(col1) # only consider rows that are not gapped

#  for (i in 1:length(colVec)) { 
#   colVec[i] <- toupper(colVec[i])
#  }
  if (is.null(ids)) {
   ids <- 1:length(colVec)
  }
  nVec <- rep(0, length(ids))
  for (i in 1:length(ids)) {
#   nVec[i] <- (count.complementary.bias(col1, colVec[i], iterations=iterations, gu.allowed=gu.allowed, keep=FALSE, pLim=pLim, iterCheck=iterCheck)[["p.value"]])
   nVec[i] <- -log(complementarity.binom.test(col1, colVec[ids[i]], gu.allowed=gu.allowed)[["p.value"]])
  }
  result <- max(nVec)
  maxId <- ids[which.max(nVec)]
  if (transform.z) {
   m <- mean(nVec)
   s <- sd(nVec)
   if (s > 0) {
    result <- max((nVec-m)/s)
   }
  }
  stopifnot(length(result) == 1)
  attr(result, "which") <- maxId
  result
}

# performs randomization test to see how unlikely it is, to observe this high z-score
count.complementary.zbias <- function(col1, colVec, ids=NULL, iterations=100, gu.allowed=TRUE, pLim=0.1, iterCheck=20, transform.z=TRUE) {
  col1 <- toupper(col1) # only consider rows that are not gapped
  nCols <- length(colVec)
#  for (i in 1:length(colVec)) { 
#   colVec[i] <- toupper(colVec[i])
#  }
  hz0  <- count.complementary.highestZ(col1, colVec, ids=ids,gu.allowed=gu.allowed, transform.z=transform.z)
#  cat("z-score of unshuffled column:", hz0, col1, "\n")
  successes <- 0
  effIterations = iterations
  conservedFlag <- FALSE
  if (is.conservedSequence(col1)) {
    effIterations = 1
    successes = 1
    conservedFlag <- TRUE
  } else {
   for (i in 1:iterations) {
     col1r <- shuffleAlignmentColumnGapPreserved(col1, collapse=TRUE)
     stopifnot(nchar(col1) == nchar(col1r))
     hz  <- count.complementary.highestZ(col1r, colVec, ids=ids, gu.allowed=gu.allowed, transform.z=transform.z)
     if (hz >= hz0) {
       successes <- successes + 1
     }
     p <- successes / i
#     cat("z-score of shuffled column:", hz, col1r, "successes:", successes,"p: ", p, "\n")
     if ((p > pLim) && (i > iterCheck)) {
      effIterations = i
      break
     }
   }
  }
  pval <- successes / effIterations
  if (successes == 0) {
    pval <- 1/effIterations
  }
  result <- list()
  result[["p.value"]] <- pval
  result[["iterations"]] <- effIterations
  result[["successes"]] <- successes
  result[["conserved"]] <- conservedFlag
  result[["number_characters"]] <- nchar(col1)
  result[["which"]] <- attr(hz0 , "which") # index
  result
}

# performs extreme value test to see how unlikely it is, to observe this high z-score
complementarity.binom.multitest <- function(col1, colVec, ids=NULL, gu.allowed=TRUE) {
  stopifnot(length(colVec) >= 1)
  col1 <- toupper(col1) # only consider rows that are not gapped
  nCols <- length(colVec)
#  for (i in 1:length(colVec)) { 
#   colVec[i] <- toupper(colVec[i])
#  }
#  hz0  <- count.complementary.highestZ(col1, colVec, ids=ids,gu.allowed=gu.allowed, transform.z=transform.z)
 if (is.null(ids)) {
   ids <- 1:length(colVec)
 }
 pVec <- rep(1, length(ids))
 for (i in 1:length(ids)) {
   pVec[i]  <- complementarity.binom.test(col1, colVec[ids[i]], gu.allowed=gu.allowed)[["p.value"]]
 }
 minP <- min(pVec)
 pVal <- 1 - ( (1-minP)^length(pVec) ) # extreme value theory
 whichId <- ids[which.min(pVec)] # which column lead to the most significant P-value?
 result <- list()
 result[["p.value"]] <- pVal
 result[["number_characters"]] <- nchar(col1)
 result[["which"]] <- whichId
 result
}

test.count.complementary.bias <- function(col1="AGUA", col2="UCAU", gu.allowed=TRUE,iterations=1000) {
  result <- count.complementary.bias(col1, col2, gu.allowed=gu.allowed, iterations=iterations)
  print(result)
  stopifnot(result[["iterations"]] == iterations)
  result
}

vectorToIdList <- function(v, value=1) {
  result <- list()
  if (!is.null(v) ) {
   for (i in 1:length(v)) {
     result[[as.character(v[i])]] <- value
   }
  }
  result
}

vectorToIdVec <- function(v, maxId, value=FALSE) {
  result <- rep(!value, maxId)
  if (!is.null(v) ) {
   for (i in 1:length(v)) {
     result[v[i]] <- value
   }
  }
  result
}

complementarity.matrixbias <- function(sequences1, sequences2=NULL, iterations=1000, gu.allowed=TRUE, pLim=0.05, complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=1, iterCheck=40, keepMatrix=TRUE, alphabet=RNAAlphabet()) {
  n <- length(sequences1)
  n1 <- nchar(sequences1[[1]]$seq)
  ary1 <- seqsToCharArray(sequences1)
  n2 <- n1
  ary2 <- ary1
  if (!is.null(sequences2)) {
   n2 <- nchar(sequences2[[1]]$seq)
   ary2 <- seqsToCharArray(sequences2)
   stopifnot(length(sequences2) == n)
  }
#  mtx <- NULL
#  if (keepMatrix) { # matrix with P-values
   mtx <- matrix(1.0, nrow=n1, ncol=n2)
#  }
  successes <- 0
  trials <- 0
  goodCombos <- list()
  goodRows <- rep(FALSE, n1)
  for (i in 1:n1) {
   s1 <- paste(ary1[,i], collapse="")
   if (is.conservedSequence(s1)) {
    next
   } else {
      goodRows[i] <- TRUE
   }
   if (verboseLevel > 1) {
    cat("Working on column", i, s1, "\n")
   }
   start2 <- 1
   if (is.null(sequences2)) {
     start2 <- i + 1
   }
   if (start2 <= n2) {
    for (j in start2:n2) {
     s2 <- paste(ary2[,j], collapse="")
#    cat("Working on second column", j, s2, "\n")
     if (!is.conservedSequence(s2)) {
      nonGapRows <- findAlphabetRows(s1, s2, alphabet)
      nonGapCount <- length(nonGapRows[nonGapRows]) 
      if ((nonGapCount >= 2) && ((nonGapCount/n) >= nonGapFractionMin)) {
       complCount <- count.complementary(s1, s2, gu.allowed=gu.allowed)      
       complFraction <- complCount / nonGapCount
       stopifnot(complFraction <= 1.0)
       if (complFraction >= complFractionMin) {
         goodCombos[[length(goodCombos)+1]] <- c(i,j)
       }
      }
     }
    }
   }
  }
  if (verboseLevel > 0) {
   cat("Finished pre-scanning. Found", length(goodCombos), "column pairs with a fraction of complementary characters greater or equal to", complFractionMin, "\n")
  }
  if (length(goodCombos) > 0) {
  for (k in 1:length(goodCombos)) {
   v <- goodCombos[[k]]
   i <- v[1]
   j <- v[2]
   s1 <- paste(ary1[,i], collapse="")
   if (verboseLevel > 1) {
    cat("Working on column", i, s1, "\n")
   }
   start2 <- 1
   if (is.null(sequences2)) {
     start2 <- i + 1
   }
   s2 <- paste(ary2[,j], collapse="")
   testResult <- count.complementary.bias(s1, s2, iterations=iterations, gu.allowed=gu.allowed, pLim = (2*pLim), iterCheck=iterCheck)
   pValue <- testResult[["p.value"]] 
   if (!is.null(mtx)) {
     if (verboseLevel > 1) {
      cat("Writing P value", pValue, "to index", i,j, "and matrix with dimensions", nrow(mtx), ncol(mtx), "\n")
     }
     mtx[i,j] <- pValue
     if (is.null(sequences2)) {
      mtx[j,i] <- mtx[i,j]
     }
   }
   isSuccess = (pValue<= pLim)
   if (verboseLevel > 1) {
    cat("Working on second column", j, s2, testResult[["p.value"]], isSuccess,"\n")
   }
   trials <- trials + 1
   if (isSuccess) {
     successes <- successes + 1
   }
  }
  }
  result <- list()
  result[["length1"]] <- n1  
  result[["length2"]] <- n2
  result[["successes"]] <- successes
  result[["trials"]] <- trials
  if (trials > 0) {
   result[["binom.test"]] <- binom.test(successes, trials, p=pLim, alternative="greater")
  } else {
   result[["binom.test"]] <- NA
  }
  result[["pLimit"]] <- pLim
  succRowCount <- 0
   trial2 <- 0
  for (i in 1:nrow(mtx)) {
    if (goodRows[i]) {
     trial2 <- trial2 + 1
     for (j in 1:ncol(mtx)) {
      if (mtx[i,j] <= pLim) {
       succRowCount <- succRowCount + 1
       break
      }
     }
    }
  }
  pRow0 = 1 - (1-pLim)^ncol(mtx) # probability of not zero successes (n failures)
  result[["binom.test.rows"]] <- binom.test(succRowCount, trial2, pRow0, alternative="greater")
  if (keepMatrix) {
    result[["matrix"]] <- mtx
  }
  result
}

RemoveConservedColumns <- function(charArray, gapAllowed=TRUE, verbose=FALSE) {
 goodIds <- c()
 for (i in 1:ncol(charArray)) {
  if (is.conservedSequence(paste(charArray[,i], collapse=""))) {
    if (verbose) {
     cat("Removing conserved sequence column", i, "\n")
    } 
  } else {
   goodIds[length(goodIds) + 1] <- i
  }
 } 
 charArray[, goodIds]
} 

# returns true, iff two columns have sufficient fraction of complementary base pairs and simultaneous non-gap characters
is.possiblyInteractingColumnPair <- function(col1, col2, gu.allowed=gu.allowed, complFractionMin=0.8, nonGapFractionMin=0.6, alphabet=RNAAlphabet()) {
  stopifnot(nchar(col1) > 0)
  stopifnot(nchar(col1) == nchar(col2))
  nonGapRows <- findAlphabetRows(col1, col2, alphabet)
  nonGapCount <- length(nonGapRows[nonGapRows]) 
  n <- nchar(col1) 
  result <- FALSE
  complCount <- NA
  complFraction <- NA 
  if ((nonGapCount >= 2) && ((nonGapCount/n) >= nonGapFractionMin)) {
       complCount <- count.complementary(col1, col2, gu.allowed=gu.allowed)      
       complFraction <- complCount / nonGapCount
       if (complFraction >= complFractionMin) {
         result <- TRUE     
       } 
  }
#  cat("is Possib. Interaction Result:", result, "nonGapCount:", nonGapCount, "complCount", complCount, "complFraction", complFraction ,
#       col1, col2, "\n")
  result
}

test.is.possiblyInteractingColumnPair <- function() {
  checkTrue(is.possiblyInteractingColumnPair("AACGU", "UUGCA"))
  checkTrue(!is.possiblyInteractingColumnPair("AACGU", "AAACA")) # no matches
  checkTrue(!is.possiblyInteractingColumnPair("AACGU---", "UUGCA---")) # too high fraction of gaps 
}

complementarity.highestZ.matrixbias <- function(sequences1, sequences2=NULL, iterations=100, gu.allowed=TRUE, pLim=0.05, verboseLevel=1, iterCheck=20, combineMethod="TPM", transform.z=TRUE, removeConservedColumns=TRUE, complFractionMin=0.8, nonGapFractionMin=0.6, pMethod="zbias") {
  selfMode <- is.null(sequences2)
  if (verboseLevel > 0) {
   if (selfMode) {
	   cat("Starting complementarity.highestZ.matrixbias with sequence lengths", nchar(sequences1[[1]]$seq),"\n")
   } else {
	   cat("Starting complementarity.highestZ.matrixbias with sequence lengths", nchar(sequences1[[1]]$seq),nchar(sequences2[[1]]$seq), "\n")
   }
  }

  n <- length(sequences1)
  ary1 <- seqsToCharArray(sequences1, transform.upper=TRUE)
  ary2 <- ary1

  if (!is.null(sequences2)) {
   ary2 <- seqsToCharArray(sequences2, transform.upper=TRUE)
   stopifnot(length(sequences2) == n)
  }
  if (removeConservedColumns) {
    ary1 <- RemoveConservedColumns(ary1, verbose=(verboseLevel> 1))
    ary2 <- RemoveConservedColumns(ary2, verbose=(verboseLevel > 1))
  }
  n1 <- ncol(ary1) # nchar(sequences1[[1]]$seq)
  n2 <- ncol(ary2) # nchar(sequences2[[1]]$seq)
  if (verboseLevel > 0) {
   cat("Running complementarity.highestZ.matrixbias with sequence lengths", n1, n2, "\n")
  }
  
  pVec <- c() # rep(1.0, n1)
  successes <- 0
  trials <- 0

   colVec <- c()
#    if (is.null(sequences2) && (j == i)) {
#     next
#    }
  for (j in 1:n2) {
   s2 <- paste(ary2[,j], collapse="")
   colVec[length(colVec) + 1] <- s2
  }
  ids <- 1:n2
  stopifnot(length(ids) == length(colVec))
  endId <- n1
  if (selfMode) {
   endId <- n1 - 2 
  }  
  stopifnot(endId >= 1)
  tabooIds <- c() # ensures that each subject column is used only once as evidence
  for (i in 1:endId) {
   s1 <- paste(ary1[,i], collapse="") # i'th column
   if (verboseLevel > 1) {
    cat("Working on column", i, s1, "\n")
   }
   if (selfMode) {
      stopifnot((i+1) <= n2)
      ids <- (i+1):n2 # only consider these columns
   }
   stopifnot(max(ids) <= length(colVec))
   stopifnot(length(ids) > 1)
   usedIds <- c()
   for (j in 1:length(ids)) {
     stopifnot(ids[j] <= length(colVec))
     if (selfMode && (ids[j] == i)) {
      stop("Internal error: alignment column should not be considered to be interacting with itself.\n")
     }
     if ((length(tabooIds) == 0) || is.na(tabooIds[ids[j]])) { # ensure that each subject column is used only once as evidence
      if (is.possiblyInteractingColumnPair(s1, colVec[ids[j]], gu.allowed=gu.allowed, complFractionMin=complFractionMin, 
                                          nonGapFractionMin=nonGapFractionMin)) {
       usedIds[length(usedIds) + 1] <- ids[j]
      }
     }
   }
   if ( ((!transform.z) && (length(usedIds) < 1)) || (transform.z && (length(usedIds) < 2))) {
      if (verboseLevel > 1) {
       cat(paste("Warning: found insufficient number of columns", i, s1, "\n")) 
      }
      next
   }
   testResult <- NULL
   if (pMethod == "zbias") {
     testResult <- count.complementary.zbias(s1, colVec, ids=usedIds, iterations=iterations, gu.allowed=gu.allowed, pLim = (2*pLim), iterCheck=iterCheck, transform.z=transform.z)
   } else if (pMethod == "binom") {
     testResult <- complementarity.binom.multitest(s1, colVec, ids=usedIds, gu.allowed=gu.allowed)
   } else {
    stop("unknown p-value computation method.\n")
   }
   stopifnot(is.numeric(testResult[["which"]]))
   pValue <- testResult[["p.value"]] 
   isSuccess = (pValue<= pLim)
   if (isSuccess) {
    tabooIds[testResult[["which"]]] <- TRUE # this subject column cannot be used again as evidence
   } 
   trials <- trials + 1
   if (verboseLevel > 1) {
    cat("Writing P value", pValue, "to index", i, "success:", isSuccess, "trials:", trials, "# cols:", length(usedIds), "which:", testResult[["which"]] , colVec[testResult[["which"]]], "\n")
   }
   pVec[length(pVec) + 1] <- pValue
   if (isSuccess) {
     successes <- successes + 1
   }
  }
  result <- list()
  result[["length1"]] <- n1  
  result[["length2"]] <- n2
  result[["successes"]] <- successes
  result[["trials"]] <- trials
  if (trials > 0) {
   result[["binom.test"]] <- binom.test(successes, trials, p=pLim, alternative="greater")
  } else {
   result[["binom.test"]] <- NA
  }
  result[["pLimit"]] <- pLim
  result[["pValues"]] <- pVec
  if (length(pVec) >= 1) { 
   result[["p.value"]] <- computeCombinedPValues(pVec, method=combineMethod)
  } else {
   result[["p.value"]] <- 1.0
  }
  result
}

ReplaceNA <- function(x, value) {
 if (is.na(x)) {
  x <- value
 }
 value
}

test.ReplaceNA <- function(x="q", value="X") {
 ReplaceNA(x,value)
}

# converts vector of characters or string to counts or fractions. Divides by number of characters, not counting - or .
# multi: multiplier such as 100 to convert it to percent

LettersToFractions <- function(chars, alphabet, multi=100, useLetterIds=NULL, rounding=TRUE) {
 if (length(chars) == 1) {
   if (is.character(chars)) {
     if (nchar(chars) > 1) {
       chars <- strsplit(chars,"")[[1]]
     }
   }
 }
 rleResult <- rle(sort(chars))
 rleHash <- rleResult[["lengths"]]
 names(rleHash) <- rleResult[["values"]]
 numGaps <- ReplaceNA(rleHash["-"], 0) + ReplaceNA(rleHash["."], 0)
 numNonHashs <- length(chars) - numGaps
 n <- length(alphabet)
 result <- rep(0, n)
 names(result) <- alphabet
 for (i in 1:n) {
   x <- rleHash[alphabet[i]]
   if (!is.na(x)) {
     result[i] <- round(multi * x / numNonHashs)
   }
 }
 if (is.null(useLetterIds)) {
   useLetterIds <- 1:(n-1) # do not use last letter
 }
 hash <- paste(result[useLetterIds], collapse="_")
 attr(result, "hash") <- hash
 result
}

test.LettersToFractions <- function(chars=c("A","C","A","A","G","G"), alphabet=c("A","C","G","U"), multi = 100) {
  result <- LettersToFractions(chars=chars, alphabet=alphabet, multi=multi)
  result
}

# hash code implementation of central method complementarity.highestZ.matrixbias 
complementarity.highestZ.matrixbias.fast <- function(sequences1, sequences2=NULL, iterations=100, gu.allowed=TRUE, pLim=0.05, verboseLevel=1, iterCheck=20, combineMethod="TPM", transform.z=TRUE, removeConservedColumns=TRUE, complFractionMin=0.8, nonGapFractionMin=0.6, pMethod="zbias", alphabet=c("A","C","G","U")) {
  selfMode <- is.null(sequences2)
  if (verboseLevel > 0) {
   if (selfMode) {
	   cat("Starting complementarity.highestZ.matrixbias with sequence lengths", nchar(sequences1[[1]]$seq),"\n")
   } else {
	   cat("Starting complementarity.highestZ.matrixbias with sequence lengths", nchar(sequences1[[1]]$seq),nchar(sequences2[[1]]$seq), "\n")
   }
  }
  n <- length(sequences1)
  ary1 <- seqsToCharArray(sequences1, transform.upper=TRUE)
  ary2 <- ary1
  if (!is.null(sequences2)) {
   ary2 <- seqsToCharArray(sequences2, transform.upper=TRUE)
   stopifnot(length(sequences2) == n)
  }
  if (removeConservedColumns) {
    ary1 <- RemoveConservedColumns(ary1, verbose=(verboseLevel> 1))
    ary2 <- RemoveConservedColumns(ary2, verbose=(verboseLevel > 1))
  }
  n1 <- ncol(ary1) # nchar(sequences1[[1]]$seq)
  n2 <- ncol(ary2) # nchar(sequences2[[1]]$seq)
  if (verboseLevel > 0) {
   cat("Running complementarity.highestZ.matrixbias with sequence lengths", n1, n2, "\n")
  }

  hashes1 <- list()
  for (i in 1:n1) {
    counts1 <- LettersToFractions(ary1[,i], alphabet)
    hashes1[[attr(counts1, "hash")]] <- append(hashes1[[attr(counts1, "hash")]], i)    
  }

  hashes2 <- list()
  for (i in 1:n2) {
    counts2 <- LettersToFractions(ary2[,i], alphabet)
    hashes2[[attr(counts2, "hash")]] <- append(hashes2[[attr(counts2, "hash")]], i)    
  }

  pVec <- c() # rep(1.0, n1)
  successes <- 0
  trials <- 0

   colVec <- c()
#    if (is.null(sequences2) && (j == i)) {
#     next
#    }
  for (j in 1:n2) {
   s2 <- paste(ary2[,j], collapse="")
   colVec[length(colVec) + 1] <- s2
  }
  ids <- 1:n2
  stopifnot(length(ids) == length(colVec))
  endId <- n1
  if (selfMode) {
   endId <- n1 - 2 
  }  
  stopifnot(endId >= 1)
  tabooIds <- c() # ensures that each subject column is used only once as evidence
  for (i in 1:endId) {
   s1 <- paste(ary1[,i], collapse="") # i'th column
   s1c <- nucleotides.complement(s1) # complement (not reverse complement)
   s1cCounts <- LettersToFractions(s1c, alphabet)
   if (length(s1cCounts[s1cCounts == 0]) >= (length(alphabet)-1)) {
    cat("Not working on conserved column", i, s1, s1c, attr(s1cCounts, "hash"), "\n")
    next
   }
   if (verboseLevel > 1) {
    cat("Working on column", i, s1, s1c, attr(s1cCounts, "hash"), "\n")
   }
   ids <- hashes2[[attr(s1cCounts, "hash")]]
   if (selfMode) {
      ids <- ids[ids > i]
   }
   if (length(ids) == 0) {
     cat("Column", i, "has no reverse complementary columns in target region\n") 
     next
   }
   stopifnot(max(ids) <= length(colVec))
   stopifnot(length(ids) >= 1)
   usedIds <- c()
   for (j in 1:length(ids)) {
     stopifnot(ids[j] <= length(colVec))
     if (selfMode && (ids[j] == i)) {
      stop("Internal error: alignment column should not be considered to be interacting with itself.\n")
     }
     if ((length(tabooIds) == 0) || is.na(tabooIds[ids[j]])) { # ensure that each subject column is used only once as evidence
      if (is.possiblyInteractingColumnPair(s1, colVec[ids[j]], gu.allowed=gu.allowed, complFractionMin=complFractionMin, 
                                          nonGapFractionMin=nonGapFractionMin)) {
       usedIds[length(usedIds) + 1] <- ids[j]
      }
     }
   }
   if ( ((!transform.z) && (length(usedIds) < 1)) || (transform.z && (length(usedIds) < 2))) {
      if (verboseLevel > 1) {
       cat(paste("Warning: found insufficient number of columns", i, s1, "\n")) 
      }
      next
   }
   testResult <- NULL
   if (pMethod == "zbias") {
     testResult <- count.complementary.zbias(s1, colVec, ids=usedIds, iterations=iterations, gu.allowed=gu.allowed, pLim = (2*pLim), iterCheck=iterCheck, transform.z=transform.z)
   } else if (pMethod == "binom") {
     testResult <- complementarity.binom.multitest(s1, colVec, ids=usedIds, gu.allowed=gu.allowed)
   } else {
    stop("unknown p-value computation method.\n")
   }
   stopifnot(is.numeric(testResult[["which"]]))
   pValue <- testResult[["p.value"]] 
   isSuccess = (pValue<= pLim)
   if (isSuccess) {
    tabooIds[testResult[["which"]]] <- TRUE # this subject column cannot be used again as evidence
   } 
   trials <- trials + 1
   if (verboseLevel > 1) {
    cat("Writing P value", pValue, "to index", i, "success:", isSuccess, "trials:", trials, "# cols:", length(usedIds), "which:", testResult[["which"]] , colVec[testResult[["which"]]], "\n")
   }
   pVec[length(pVec) + 1] <- pValue
   if (isSuccess) {
     successes <- successes + 1
   }
  }
  result <- list()
  result[["length1"]] <- n1  
  result[["length2"]] <- n2
  result[["successes"]] <- successes
  result[["trials"]] <- trials
  if (trials > 0) {
   result[["binom.test"]] <- binom.test(successes, trials, p=pLim, alternative="greater")
  } else {
   result[["binom.test"]] <- NA
  }
  result[["pLimit"]] <- pLim
  result[["pValues"]] <- pVec
  if (length(pVec) >= 1) { 
   result[["p.value"]] <- computeCombinedPValues(pVec, method=combineMethod)
  } else {
   result[["p.value"]] <- 1.0
  }
  result
}

test.count.complementary.matrixbias <- function(iterations=100, gu.allowed=TRUE,pLim=0.05, shuffle=FALSE, verboseLevel=2, keepMatrix=FALSE, startPos=1, stopPos=70) {
  file1 = system.file("data/FBgn0011872.fa", package="rnafolding") # system.file("data/FBgn0011872.fa", package="rnafolding")
#  file2 = "../data/FBgn0011859.fa" # , package="rnafolding")
  stopifnot(file.exists(file1))
#  stopifnot(file.exists(file2))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  seqs1 <- extractAlignment(seqs1, startPos, stopPos)
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
#  seqs2 <- readFASTA(file2, strip.descs=TRUE)
  result <- count.complementary.matrixbias(seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, keepMatrix=keepMatrix)
  result
}

test.complementarity.matrixbias.anticodon <- function(iterations=100, gu.allowed=TRUE,pLim=0.05, shuffle=FALSE, verboseLevel=2, keepMatrix=FALSE, startPos=1, stopPos=70) {
  file1 = system.file("data/RNA_ecoli_anticodon.fa", package="rnafolding") # system.file("data/FBgn0011872.fa", package="rnafolding")
#  file2 = "data/FBgn0011859.fa" # , package="rnafolding")
  stopifnot(file.exists(file1))
#  stopifnot(file.exists(file2))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  seqs1 <- extractAlignment(seqs1, startPos, stopPos)
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
#  seqs2 <- readFASTA(file2, strip.descs=TRUE)
  result <- count.complementary.matrixbias(seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, keepMatrix=keepMatrix)
  result
}

# checks if D-loop of tRNA is classified as "structured"
test.complementarity.highestZ.matrixbias <- function(iterations=50, gu.allowed=TRUE,pLim=0.05, shuffle=FALSE, verboseLevel=2, keepMatrix=FALSE, region=range(10,30), transform.z=TRUE, pMethod="zbias", fast=FALSE) {
  file1 = system.file("data/FBgn0011872.fa", package="rnafolding") # system.file("data/FBgn0011872.fa", package="rnafolding")
#  file2 = "data/FBgn0011859.fa" # , package="rnafolding")
  stopifnot(file.exists(file1))
#  stopifnot(file.exists(file2))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  if (!is.null(region)) {
   seqs1 <- extractAlignment(seqs1, region[1], region[2])
  }
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
#  seqs2 <- readFASTA(file2, strip.descs=TRUE)
  if (fast) {
    result <- complementarity.highestZ.matrixbias.fast(seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z)
  } else {
    result <- complementarity.highestZ.matrixbias(seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z)
  }

  result
}

# checks if D-loop of tRNA is classified as "structured"
test.complementarity.highestZ.matrixbias.anticodon <- function(iterations=200, gu.allowed=TRUE,pLim=0.05, shuffle=FALSE, verboseLevel=2, region=NULL, transform.z=FALSE, pMethod="zbias", complFractionMin=0.8, nonGapFractionMin=0.6, fast=FALSE) {
  file1 = system.file("data/tRNA_ecoli_anticodon.fa", package="rnafolding")
#  file2 = "../data/FBgn0011859.fa" # , package="rnafolding")
  stopifnot(file.exists(file1))
#  stopifnot(file.exists(file2))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  if (!is.null(region)) {
   seqs1 <- extractAlignment(seqs1, region[1], region[2])
  }
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
#  seqs2 <- readFASTA(file2, strip.descs=TRUE)
  if (!fast) {
    result <- complementarity.highestZ.matrixbias(seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  } else {
    result <- complementarity.highestZ.matrixbias.fast(seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  }
  result
}

test.complementarity.highestz.matrixbias.gapped <- function(file1=system.file("data/iicand1a.fa", package="rnafolding"), file2=system.file("data/iicand1b.fa", package="rnafolding"), pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, fast=FALSE) {
	seqs1 <- readFASTA(file1, strip.descs=TRUE)
	seqs2 <- readFASTA(file2, strip.descs=TRUE)
        if (fast) {
   	 result <- complementarity.highestZ.matrixbias.fast(seqs1, sequences2=seqs2, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
        } else { 
   	 result <- complementarity.highestZ.matrixbias(seqs1, sequences2=seqs2, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
        }
        result
}

test.complementarity.highestz.matrixbias.cand2 <- function(file1="../data/iicand2a.fa", file2="../data/iicand2b.fa", pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, fast=FALSE) {
	seqs1 <- readFASTA(file1, strip.descs=TRUE)
	seqs2 <- readFASTA(file2, strip.descs=TRUE)
        if (fast) {
  	 result <- complementarity.highestZ.matrixbias.fast(seqs1, sequences2=seqs2, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
        } else {
  	 result <- complementarity.highestZ.matrixbias(seqs1, sequences2=seqs2, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
        }
        result
}

# checks if D-loop of tRNA is classified as "structured"
test.complementarity.highestZ.matrixbias.2 <- function(iterations=200, gu.allowed=TRUE,pLim=0.05, shuffle=FALSE, verboseLevel=2, region1=NULL, region2=NULL, transform.z=TRUE,
                          fast=FALSE) {
  file1 = system.file("data/FBgn0011872.fa", package="rnafolding") #  system.file("data/FBgn0011872.fa", package="rnafolding")
  file2 = system.file("data/FBgn0011859.fa", package="rnafolding")
  stopifnot(file.exists(file1))
  stopifnot(file.exists(file2))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  if (!is.null(region1)) {
   seqs1 <- extractAlignment(seqs1, region1[1], region1[2])
  }
  if (!is.null(region2)) {
   seqs2 <- extractAlignment(seqs2, region2[1], region2[2])
  }
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
  seqs2 <- readFASTA(file2, strip.descs=TRUE)
  if (fast) {
    result <- complementarity.highestZ.matrixbias.fast(seqs1, seqs2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, transform.z=transform.z, verboseLevel=verboseLevel)
  } else {
    result <- complementarity.highestZ.matrixbias(seqs1, seqs2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, transform.z=transform.z, verboseLevel=verboseLevel)
  }

  result
}

test.count.complementary.matrixbias2 <- function(iterations=100, gu.allowed=TRUE,pLim=0.05, shuffle=FALSE, verboseLevel=2, keepMatrix=FALSE) {
  file1 = system.file("data/FBgn0011872.fa", package="rnafolding") # system.file("data/FBgn0011872.fa", package="rnafolding")
  file2 = system.file("data/FBgn0011859.fa", package="rnafolding")
  stopifnot(file.exists(file1))
  stopifnot(file.exists(file2))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  seqs2 <- readFASTA(file2, strip.descs=TRUE)
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
   seqs2 <- shuffleSequencesGapPreserved(seqs2)
  }
  result <- count.complementary.matrixbias(seqs1, seqs2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, keepMatrix=keepMatrix)
  result
}

# convenience function: wrapper for file1 and file2
run.count.complementary.matrixbias <- function(file1, file2=NULL, iterations=100, gu.allowed=TRUE, pLim=0.05, shuffle=FALSE, verboseLevel=0) {
  stopifnot(file.exists(file1))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  seqs2 <- NULL
  if (!is.null(file2)) {
   stopifnot(file.exists(file2))
   seqs2 <- readFASTA(file2, strip.descs=TRUE)
  }
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
  result <- count.complementary.matrixbias(seqs1, seqs2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, keepMatrix=FALSE)
#  print(result)
#  cat("more:\n")
  tresult <- result[["binom.test"]]
  pval <- 1
  stat <- NA
  param <- NA
  estim <- NA
  if (!is.na(tresult)) {
   pval <- tresult[["p.value"]]
   stat <- tresult[["statistic"]]
   param <- tresult[["parameter"]]
   estim <- tresult[["estimate"]]
  }
  cat(pval, stat, param, estim, "\n")
#  result
}


# convenience function: wrapper for file1 and file2 and function complementarity.highestZ.matrixbias
run.complementarity.highestZ.matrixbias <- function(file1, file2=NULL, iterations=200, gu.allowed=TRUE, pLim=0.05, shuffle=FALSE, verboseLevel=0, transform.z=FALSE, pMethod="binom", matrixFile=NULL, reverse1=FALSE, reverse2=FALSE, complFractionMin=0.8, nonGapFractionMin=0.6, outputSep="\n", compute.diag=TRUE, combine.p.method="tpms") {
  stopifnot(file.exists(file1))
  seqs1 <- readFASTA(file1, strip.descs=TRUE)
  seqs1 <- ConvertDNAtoRNAAlignment(seqs1)
  seqs2 <- NULL
  if (!is.null(file2)) {
   stopifnot(file.exists(file2))
   seqs2 <- readFASTA(file2, strip.descs=TRUE)
   seqs2 <- ConvertDNAtoRNAAlignment(seqs2)
  }
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
  if (reverse1) {
    seqs1 <- alignment.reverseComplement(seqs1, dna=FALSE) # in RNA mode
  }
  if (reverse2) {
    if (!is.null(seqs2)) {
     seqs2 <- alignment.reverseComplement(seqs2, dna=FALSE) # in RNA mode
    }
  }
  result <- complementarity.highestZ.matrixbias(seqs1, seqs2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  pVal <- result[["p.value"]]
  pComb <- pVal
  pDiag <- 1
  if (compute.diag) {
   mtx <- complementarity.binom.test.matrix(seqs1, seqs2, gu.allowed=gu.allowed, verbose=FALSE)
   if (!is.null(matrixFile)) {
   	write.matrix(mtx, file=matrixFile)
   }
   pDiag <- diagBias(-log(mtx), thresh=-log(pLim))$p.value
 #  print(result)
 #  cat("more:\n")
   # tresult <- result[["binom.test"]]
   pComb <- computeCombinedPValues(c(pVal, pDiag), method=combine.p.method)  # combine two independent P-values
  }
  if (verboseLevel >= 0) {
   cat(pVal, " ",pDiag, " ", pComb, outputSep, sep="")
  }
  result[["p.value.diag"]] <- pDiag
  result[["p.value.comb"]] <- pComb
  result
}

# convenience function: wrapper for file1 and file2 and function complementarity.highestZ.matrixbias
multirun.complementarity.highestZ.matrixbias <- function(file1, idv1=NULL, idv2=NULL, iterations=200, gu.allowed=TRUE, pLim=0.05, shuffle=FALSE, verboseLevel=0, transform.z=FALSE, pMethod="binom", matrixFile=NULL, reverse1=FALSE, reverse2=FALSE, complFractionMin=0.8, nonGapFractionMin=0.6, outputSep="\n", compute.diag=FALSE, combine.p.method="tpms") {

  ids1 <- c()
  ids2 <- c()
  pVals <- c()  
  pVals1 <- c()  
  pVals2 <- c()  
  pDiags <- c()
  pCombs <- c()

  mseqs1 <- file1
  if (is.character(file1)) {
   stopifnot(file.exists(file1))
   mseqs1 <- readMultiFASTA(file1)
  } 
#  mseqs1 <- ConvertDNAtoRNAAlignment(seqs1)
#  mseqs2 <- NULL
#  if (!is.null(file2)) {
#   stopifnot(file.exists(file2))
#   mseqs2 <- readFASTA(file2, strip.descs=TRUE)
#   seqs2 <- ConvertDNAtoRNAAlignment(seqs2)
#  }
  for (i in 1:length(mseqs1) ) {
    mseqs1[[i]] <- ConvertDNAtoRNAAlignment(mseqs1[[i]])
  }
  if (is.null(idv1)) {
   idv1 <- 1:length(mseqs1)
  }
  for (i1 in idv1 ) {
   seqs1 <- mseqs1[[i1]]

  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
  if (reverse1) {
    seqs1 <- alignment.reverseComplement(seqs1, dna=FALSE) # in RNA mode
  }
  if (is.null(idv2)) {
   idv2 <- i1:length(mseqs1)
  }
  for (i2 in idv2) {
   seqs2 <- mseqs1[[i2]]
  if (reverse2) {
    if (!is.null(seqs2)) {
     seqs2 <- alignment.reverseComplement(seqs2, dna=FALSE) # in RNA mode
    }
  }
  result1 <- complementarity.highestZ.matrixbias(seqs1, seqs2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  # opposite order:
  result2 <- complementarity.highestZ.matrixbias(seqs2, seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  pVal1 <- result1[["p.value"]]
  pVal2 <- result2[["p.value"]]
  pVal <- max(pVal1,pVal2) # conservative estimate of combined P-value
  pComb <- pVal
  pDiag <- 1
  if (compute.diag) {
   mtx <- complementarity.binom.test.matrix(seqs1, seqs2, gu.allowed=gu.allowed, verbose=FALSE)
   pDiag <- diagBias(-log(mtx), thresh=-log(pLim))$p.value
 #  print(result)
 #  cat("more:\n")
   # tresult <- result[["binom.test"]]
   pComb <- computeCombinedPValues(c(pVal, pDiag), method=combine.p.method)  # combine two independent P-values
  }
  if (verboseLevel >= 0) {
   cat(i1," ", i2," ",pVal, " ",pDiag, " ", pComb, " ", pVal1, " ", pVal2, outputSep, sep="")
  }
#  result[["p.value.diag"]] <- pDiag
#  result[["p.value.comb"]] <- pComb
   ids1 <- append(ids1, i1)
   ids2 <- append(ids2, i2)
   pVals <- append(pVals, pVal)
   pVals1 <- append(pVals1, pVal1)
   pVals2 <- append(pVals2, pVal2)
   pDiags <- append(pDiags, pDiag)
   pCombs <- append(pCombs, pComb)
   }
  } 
  data.frame(cbind(ID1=ids1,ID2=ids2,P=pVals,P1=pVals1,P2=pVals2 ,Pd=pDiags,Pc=pCombs))
}

test.run.complementarity.highestz.matrixbias.gapped <- function(file1=system.file("data/iicand1a.fa", package="rnafolding"), file2=system.file("data/iicand1b.fa", package="rnafolding"), pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, reverse1=FALSE, reverse2=FALSE) {
   result <- run.complementarity.highestZ.matrixbias(file1, file2, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, reverse1=reverse1, reverse2=reverse2)
   result
}

test.multirun.complementarity.highestz.matrixbias <- function(file1=system.file("data/iicand1a.fa", package="rnafolding"), file2=system.file("data/iicand1b.fa", package="rnafolding"), idv1=NULL,pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, reverse1=FALSE, reverse2=FALSE, compute.diag=FALSE) {
   f1 <- readFASTA(file1, strip.descs=TRUE)
   f2 <- readFASTA(file2, strip.descs=TRUE) 
   fc <- list()
   fc[[1]] <- f1
   fc[[2]] <- f2
   result <- multirun.complementarity.highestZ.matrixbias(fc, idv1=idv1, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, reverse1=reverse1, reverse2=reverse2, compute.diag=compute.diag)
   result
}

# reads first three aligment corresponding to exons and performs all-versus-all analysis
test.multirun.complementarity.highestz.matrixbias.2 <- function(file=system.file("data/dm3_A27_ds2_exons_top3.fasta", package="rnafolding"), idv1=NULL,pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, reverse1=FALSE, reverse2=FALSE, compute.diag=FALSE) {
   result <- multirun.complementarity.highestZ.matrixbias(file, idv1=idv1, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, reverse1=reverse1, reverse2=reverse2, compute.diag=compute.diag)
   result
}

# reads 37 aligments corresponding to exons and performs all-versus-all analysis
test.multirun.complementarity.highestpmu.matrixbias.3 <- function(file=system.file("data/dm3_A27_allexons_chrM.fasta", package="rnafolding"), idv1=NULL,pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, reverse1=FALSE, reverse2=FALSE, compute.diag=FALSE) {
   result <- multirun.complementarity.highestZ.matrixbias(file, idv1=idv1, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, reverse1=reverse1, reverse2=reverse2, compute.diag=compute.diag)
   result
}

# convenience function: wrapper for file1 and file2 and function complementarity.highestZ.matrixbias
run.complementarity.highestZ.matrixbias.both <- function(file1, file2=NULL, iterations=200, gu.allowed=TRUE, pLim=0.05, shuffle=FALSE, verboseLevel=0, transform.z=FALSE, pMethod="binom", matrixFile=NULL, complFractionMin=0.8, nonGapFractionMin=0.6, compute.diag=TRUE) {
   result <- list() 
   result[["original"]] <- run.complementarity.highestZ.matrixbias(file1=file1, file2=file2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, shuffle=shuffle, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, matrixFile=matrixFile, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, reverse1=FALSE, reverse2=FALSE, outputSep=" ", compute.diag=compute.diag)
   result[["reverse_complement"]] <- run.complementarity.highestZ.matrixbias(file1=file1, file2=file2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, shuffle=shuffle, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, matrixFile=matrixFile, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, reverse1=TRUE, reverse2=TRUE, outputSep="\n", compute.diag=compute.diag)
  result
}

# convenience function: wrapper for file1 and file2 and function complementarity.highestZ.matrixbias; 
# both files are expected to be in "multi-FASTA" format with same number of alignments
pmultirun.complementarity.highestZ.matrixbias <- function(file1, file2, idRange=NULL, iterations=200, gu.allowed=TRUE, pLim=0.05, shuffle=FALSE, verboseLevel=0, transform.z=FALSE, pMethod="binom", matrixFile=NULL, complFractionMin=0.8, nonGapFractionMin=0.6, outputSep="\n", compute.diag=FALSE, combine.p.method="tpms", antidiag=TRUE, symmetric.triangle.only=FALSE, seqVec=NULL,votingMode=FALSE) {

  ids1 <- c()
  ids2 <- c()
  pVals <- c()  
  pValsB <- c()  
  pVals1 <- c()  
  pVals2 <- c()  
  pVals1B <- c()  
  pVals2B <- c()  
  pDiags <- c()
  pDiagsRC <- c()
  pCombs <- c()
  pCombsB <- c()
  dnaOrRnaVec1 <- c()
  dnaOrRnaVec2 <- c()
  mseqs1 <- file1
  if (is.character(file1)) {
   stopifnot(file.exists(file1))
   if (is.null(idRange)) {
    mseqs1 <- readMultiFASTA(file1)
   } else {
    mseqs1 <- readMultiFASTA(file1, firstBlock=min(idRange), lastBlock=max(idRange))
   }
  } 
#  mseqs1 <- ConvertDNAtoRNAAlignment(seqs1)
  mseqs2 <- file2
  if (is.character(file2)) {
   stopifnot(file.exists(file2))
   if (is.null(idRange) ) {
    mseqs2 <- readMultiFASTA(file2)
   } else {
    mseqs2 <- readMultiFASTA(file2, firstBlock=min(idRange), lastBlock=max(idRange))
   }
  } 
  if (length(mseqs1) != length(mseqs2)) {
    stop(paste("Number of alignments must be equal:", length(mseqs1), length(mseqs2), "\n"))
  }
  for (i in 1:length(mseqs1) ) {
    mseqs1[[i]] <- ConvertDNAtoRNAAlignment(mseqs1[[i]]) # convert to RNA
    mseqs2[[i]] <- ConvertDNAtoRNAAlignment(mseqs2[[i]])
    if (!is.null(seqVec)) { # use only subset of sequences
     mseqs1[[i]] <- mseqs1[[i]][seqVec]
     mseqs2[[i]] <- mseqs2[[i]][seqVec]
    }
  }

  idv <- 1:length(mseqs1)

  for (i1 in idv ) {
   seqs1 <- mseqs1[[i1]]
  if (shuffle) {
   seqs1 <- shuffleSequencesGapPreserved(seqs1)
  }
  i2 <- i1
  seqs2 <- mseqs2[[i2]]
  result1 <- complementarity.highestZ.matrixbias(seqs1, seqs2, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  # opposite order:
  result2 <- complementarity.highestZ.matrixbias(seqs2, seqs1, iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  pVal1 <- result1[["p.value"]]
  pVal2 <- result2[["p.value"]]
  pVal <- max(pVal1,pVal2) # conservative estimate of combined P-value

  # same idea, but using reverse-complement of both alignments:
  result1b <- complementarity.highestZ.matrixbias(alignment.reverseComplement(seqs1), alignment.reverseComplement(seqs2), iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
  # opposite order (not necessary):
  result2b <- complementarity.highestZ.matrixbias(alignment.reverseComplement(seqs2), alignment.reverseComplement(seqs1), iterations=iterations, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=transform.z, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin)
#  pValB <- result1b[["p.value"]]
  pVal1B <- result1b[["p.value"]]
  pVal2B <- result2b[["p.value"]]
  pValB <- max(pVal1B,pVal2B) # conservative estimate of combined P-value

  pComb <- pVal
  pCombB <- pValB
  pDiag <- 1
  pDiagRO <- 1
  pDiagB <- 1
  pDiagBRO <- 1
  dnaOrRna1 <- NA
  dnaOrRna2 <- NA
  if (compute.diag) {
   mtx <- complementarity.binom.test.matrix(seqs1, seqs2, gu.allowed=gu.allowed, verbose=FALSE)
   pDiag <- diagBias(-log(mtx), thresh=-log(pLim), symmetric.triangle.only=symmetric.triangle.only, antidiag=antidiag)$p.value
   mtxB <- complementarity.binom.test.matrix(alignment.reverseComplement(seqs1), alignment.reverseComplement(seqs2), gu.allowed=gu.allowed, verbose=FALSE)
   if (!is.null(matrixFile)) {
    write.matrix(mtx, file=matrixFile)
    write.matrix(mtxB, file=paste(matrixFile, "2", sep=".")) # write file corresponding to reverse complement
   }
   pDiagB <- diagBias(-log(mtxB), thresh=-log(pLim), symmetric.triangle.only=symmetric.triangle.only, antidiag=antidiag)$p.value
   #  print(result)
   #  cat("more:\n")
   # tresult <- result[["binom.test"]]
   pComb <- computeCombinedPValues(c(pVal, pDiag), method=combine.p.method)  # combine two independent P-values
   pCombB <- computeCombinedPValues(c(pValB, pDiagB), method=combine.p.method)  # combine two independent P-values
   dnaOrRna1 <- DNAorRNA.votes.matrix(mtx, seqs1, seqs2, pLimCons=pLim, pLimCol=(2*pLim), selfMode=FALSE)[["consensus"]]
   dnaOrRna2 <- DNAorRNA.votes.matrix(mtxB, alignment.reverseComplement(seqs1), alignment.reverseComplement(seqs2),
  							 pLimCons=pLim, pLimCol=(2*pLim), selfMode=FALSE)[["consensus"]]
   stopifnot(is.character(dnaOrRna1))
   stopifnot(is.character(dnaOrRna2))
  }
  if (verboseLevel > 0) {
   cat("ID: ", i1," "," pVal: ",pVal, " pValB: ", pValB, " pDiag: ",pDiag," pDiagB:", pDiagB, " pComb: ", pComb, " pCombB: ", pCombB, outputSep, sep="")
#   cat("ID: ", i1," "," pVal: ",pVal, " pValB: ", pValB, " pDiag: ",pDiag," pDiagRO ",pDiagRO," pDiagB:", pDiagB," pDiagBRO ",pDiagBRO, " pComb: ", pComb, " pCombB: ", pCombB, outputSep, sep="")
  }
#  result[["p.value.diag"]] <- pDiag
#  result[["p.value.comb"]] <- pComb
   pVals <- append(pVals, pVal)
#   pVals1 <- append(pVals1, pVal1)
#   pVals2 <- append(pVals2, pVal2)
   pDiags <- append(pDiags, pDiag)
   pDiagsRC <- append(pDiagsRC, pDiagB)
   pCombs <- append(pCombs, pComb)

# for reverse complement of both alignments:
   pValsB <- append(pValsB, pValB)
#   pVals1B <- append(pVals1B, pVal1B)
#   pVals2B <- append(pVals2B, pVal2B)
   pCombsB <- append(pCombsB, pCombB)
   dnaOrRnaVec1 <- append(dnaOrRnaVec1, dnaOrRna1)
   dnaOrRnaVec2 <- append(dnaOrRnaVec2, dnaOrRna2)
  }
  idOffs <- 0
  if (!is.null(idRange)) {
   idOffs <- min(idRange) - 1
  }
  idv2 <- idv + idOffs
  finalResult <- NULL 
  if (votingMode) {
    finalResult <- data.frame(cbind(ID=idv2, P=pVals, Prc=pValsB,Pd=pDiags,Pdrc=pDiagsRC,Pc=pCombs, Pcrc=pCombsB, DNAorRNA1=dnaOrRnaVec1, DNAorRNA2=dnaOrRnaVec2))
  } else {
    finalResult <- data.frame(cbind(ID=idv2, P=pVals, Prc=pValsB,Pd=pDiags,Pdrc=pDiagsRC,Pc=pCombs, Pcrc=pCombsB))
  }
  finalResult
}

# reads first three aligment corresponding to exons and performs all-versus-all (with same id) analysis
test.pmultirun.complementarity.highestz.matrixbias.2 <- function(file=system.file("data/dm3_A27_ds2_exons_top3.fasta", package="rnafolding"), idRange=NULL,pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, compute.diag=FALSE) {
   result <- pmultirun.complementarity.highestZ.matrixbias(file, file, idRange=idRange, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, compute.diag=compute.diag)
   result
}

# reads two aligments corresponding to exons and performs analysis
test.pmultirun.complementarity.highestz.matrixbias <- function(file1=system.file("data/iicand1a.fa", package="rnafolding"), file2=system.file("data/iicand1b.fa", package="rnafolding"), idRange=NULL,pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, compute.diag=TRUE) {
   result <- pmultirun.complementarity.highestZ.matrixbias(file1, file2, idRange=idRange, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, compute.diag=compute.diag)
   result
}

# reads two alignments. Checks, if rerunning in reverse order leads to approximately same result
#  dm3_A24_ds47_chr42_top1.fasta dm3_A24_ds33_chr4_top1.fasta
test.pmultirun.complementarity.highestz.matrixbias.3 <- function(file1=system.file("data/dm3_A24_ds33_chr4_top1.fasta",package="rnafolding"),
                                                                 file2=system.file("data/dm3_A24_ds47_chr42_top1.fasta",package="rnafolding"),
  idRange=NULL,pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, compute.diag=TRUE) {
   result <- pmultirun.complementarity.highestZ.matrixbias(file1, file2, idRange=idRange, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, compute.diag=compute.diag)
  print(result) 
  p1 <- result[1,"P"]
   result2 <- pmultirun.complementarity.highestZ.matrixbias(file2, file1, idRange=idRange, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, compute.diag=compute.diag)
 print(result2)
  p2 <- result2[1,"P"]
  if (p1 != p2) {
    warning(paste("The P-values seem to be different for different orders\n", p1,p2, file1, file2))
  }
 finalResult <- list()
 finalResult[[1]] <- result
 finalResult[[2]] <- result2 
 finalResult
}


# reads two alignments. Checks, if rerunning in reverse order leads to approximately same result
#  dm3_A24_ds47_chr42_top1.fasta dm3_A24_ds33_chr4_top1.fasta
test.pmultirun.complementarity.highestz.matrixbias.tRNA <- function(file1=system.file("data/tRNA_ecoli_anticodon.fa", package="rnafolding"),
                                                                 file2=system.file("data/tRNA_ecoli_anticodon.fa", package="rnafolding"),
  idRange=NULL,pMethod="binom", complFractionMin=0.8, nonGapFractionMin=0.6, verboseLevel=2, gu.allowed=TRUE, pLim=0.05, compute.diag=TRUE) {
   result <- pmultirun.complementarity.highestZ.matrixbias(file1, file2, idRange=idRange, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, compute.diag=compute.diag)
  print(result) 
  p1 <- result[1,"P"]
   result2 <- pmultirun.complementarity.highestZ.matrixbias(file2, file1, idRange=idRange, gu.allowed=gu.allowed, pLim=pLim, verboseLevel=verboseLevel, transform.z=FALSE, pMethod=pMethod, complFractionMin=complFractionMin, nonGapFractionMin=nonGapFractionMin, compute.diag=compute.diag)
 print(result2)
  p2 <- result2[1,"P"]
  if (p1 != p2) {
    warning(paste("The P-values seem to be different for different orders\n", p1,p2, file1, file2))
  }
 finalResult <- list()
 finalResult[[1]] <- result
 finalResult[[2]] <- result2 
 finalResult
}
