// --*- C++ -*------x---------------------------------------------------------
// $Id:
//
// Program:         - 
//
// Author:          Eckart Bindewald
//
// Project name:    -
//
// Date:            $Date: 2005/11/14 16:02:03 $
//
// Description:     
// 
// -----------------x-------------------x-------------------x-----------------

#include <iostream>
#include <iomanip>
#include <string>
#include <Vec.h>
#include <debug.h>
// #include <GetArg.h>
// #include <FileName.h>
#include <Alignment.h>
// #include <CompensationScorer.h>
#include <Vector3D.h>
// #include <vectornumerics.h>
#include <aligncolor_help.h>
#include <colorFunctions.h>
#include <algorithm>
#include <stemhelp.h>
#include <Stem.h>
#include <ResourceBundle.h>
#include <Vrml2MatrixWriter.h>
#include <JVXMatrixWriter.h>
#include <Letter3D.h>
#include <AlignColorConstants.h>
// #include <SimpleCompensationMatrixScorer.h>
#include <correlogoConstants.h>
#include <CorrectedMutualInformationScorer.h>
#include <MutualInformationScorer.h>
#include <pdbtools.h>

const char SLASH = '/';
const string SLASH_STR = "/";
const string PROGRAM_VERSION = "1.2.6";

/*
const int RENDER_ENTROPY = 2;
const int RENDER_MATRIX = 4;

const int VRML1_FORMAT = 1;
const int VRML2_FORMAT = 2;
const int JVX_FORMAT = 3;
*/
void
helpOutput(ostream& os)
{
  os << "Correlogo version " << PROGRAM_VERSION << endl;
  os << "Correlogo computes 3D representations containing the mutual information corresponding to nucleotide sequence alignments." << endl;
  os << "For a given input of a specified file with the sequence alignment, the program generates 3 output files for viewing with " << endl
     << " a VRML viewer (file ending .wrl), the JavaView software (file ending .jvx, see http://www.javaview.de) or a Text editor/web browser (ending .txt)" << endl;
  os << "usage: correlogo -i <FASTA_FILE> -b <PARAMETER_FILE>" << endl   
      << "The mandatory option -i speficies the input file containing aligned nucleotide sequences in FASTA format." << endl;
  os << "The option -b <FILENAME> specifies the location of the parameter needed for generating the graphics. If the option is not present," << endl
     << "the program attempts to read the file $CORRELOGO_HOME/resources/correlogo_bitmaps.dat . In other words, if the environment variable" << endl
     << "CORRELOGO_HOME is set to the installation directory of the correlogo software, it is not necessary to specify the option -b" << endl;
  os << "Advanced options:" << endl
     << "--arrow mode : 0 : no arrow, 1: draw arrow" << endl
     << "-b parameterfilename : filename of parameter data containing the coordinates of the nucleotide letters A,C,G,U,T" << endl
     << "--bitmaps parameterfilename : synonymous to option -b" << endl
     << "-c colormode " << endl
     << "--collapse id    : collapse alignment with respect to sequence n" << endl
     << "--combo mode     : sets several options at once. Currently defined mode: 3d (equivalent to --arrow 0 --gaps 0 --logo 0)" << endl
     << "--error-mode errormode : 0: switched off, 1: show error bars" << endl
     << "--gaps gapmode  : 1: default, 0 : do not draw gap bars" << endl
     << "-l value      : threshold for mutual information values to be displayed" << endl
     << "-lr value     : threshold (relative number of standard deviations) for mutual information values to be displayed" << endl
     << "-lm value     : matrix minimum values" << endl
     << "--letter mode  : set letter mode in sequence logo. 0: no letters, 1: built-in characters 3: texture mapping" << endl
     << "--letter2 mode  : set letter mode in structure logo. 0: no letters, 1: built-in characters 3: texture mapping" << endl
     << "-m matrixfile"
     << "-o filename   : output vrml filename" << endl
     << "-r filename   : reference stems (defined format with --rf 1|2|3" << endl
     << "--reverse 0|1" << endl
     << "--stems filename  : file with stems to be displayed as stack" << endl
     << "--type RNA|DNA    : type of sequence" << endl
     << "--rf 1|2|3    : format of reference stems (region file, bracket, matrix)" << endl
     << "--verbose 0|1|2 : verbose level (0: least verbose 2: most verbose)" << endl
     << "--www URL   : url base for texture mapping images" << endl
     << endl;
}

/** output of command line parameter with which the program was called. */
void
parameterOutput(ostream& os, int argc, char** argv)
{
  for (int i = 0; i < argc; i++)
    {
      os << argv[i] << " ";
    }
  os << endl;
}

// add a slash to end of string if not already there 
void
addSlash(string & s)
{
  if (s.size() == 0) {
    return;
    // s = "./";
  }
  if (s[s.size()-1] != '/') {
    s = s + "/";
  }
}

// add path to non absolute filename
void addPathIfRelative(string& s, const string& path)
{
  if (s.size() < 1) {
    return;
  }
  string p = path;
  if ( (s[0] != '/') && (s[0] != '.') ){
    addSlash(p);
    s = p + s;
  }
}

string fileNameBase( const string &fileName )
{
  string fileNameBase, tempFileName;

  tempFileName = fileName;
  // first check, if there is a path, that means, if there is the
  // letter '/':
  if( tempFileName.find_last_of('/') < tempFileName.size() )
    {
      // remove the path:
#ifdef G_COMPILER
      tempFileName.remove( 0, (tempFileName.find_last_of('/')+1) ); 
#else
      tempFileName.erase( 0, (tempFileName.find_last_of('/')+1) ); 
#endif
      fileNameBase = tempFileName;
    }
  // then check, if there is an ending, that means, if there is the
  // letter '.'
  if( tempFileName.find_last_of('.') < tempFileName.size() )
    {
      // now check, if the point is not part of the path, therefore
      // look first, IF there exists an path and then, If this path is
      // written with a point:
      if( (tempFileName.size() < tempFileName.find_last_of('/')) ||
	  (tempFileName.find_last_of('/') < tempFileName.find_last_of('.')) )
	{
#ifdef G_COMPILER
	  tempFileName.remove( tempFileName.find_last_of('.') );  
#else
	  tempFileName.erase( tempFileName.find_last_of('.') ); 
#endif
	}
    }
  fileNameBase = tempFileName;
  return fileNameBase;
}

void 
searchValueOfOption( char option[80], string &input, char *argv )
{
  // if there's an option, get this option:
  if( argv[0] == '-' ) 
    {
      // if the next character is a "-", you know, that the option
      // is a word, otherwise it is one character:
      if( argv[1] == '-' ) 
	{
	  int k = 0;
	  while( argv[k] != '\0' ) { 
	    option[k] = argv[k+1];
	    ++k;
	  }
	  input = "";
	} // if( argv[1] == '-' )
      else
	{

	  // no option must start with a number 
	  if ( ( argv[1] >= '0' ) && (argv[1] <= '9' ) )
	    { 
	      input = argv; 
	    }
	  else
	    { 
	      option[0] = argv[1]; 
	      option[1] = '\0';
	      // if the option contains only one character, check, if
	    // the value is appended directly:
	      if( argv[2] != '\0' )
		{
		  char tempName[80];
		  
		  int k = 1;
		  while( argv[k] != '\0' ) { 
		    tempName[k-1] = argv[k+1]; 
		    ++k;
		  }
		  input = tempName;
		}
	      else
		{ input = ""; }
	    } // else (if(argv[1]=='-') ) 
	} // if( argv[0] == '-' )
    }  
  else
    { input = argv; }
}

/** return true, if option is present in command line */
bool 
isPresent(const string& origOption, int argc, char** argv)
{
  ERROR_IF(origOption=="", "Empty string for option given.", exception);
  string option = origOption;
  option = "-" + option;
  for (int i = 1; i < argc; i++)
    {
      if (option.compare(argv[i])==0)
	{
	  return true;
	}
    }

  return false; // option not found
}

// read stream, return parameters like argc and argv
char **
streamToCommands( istream& is, int& argc, const string& startString)
{
  // get vector of words
  string line;
  vector<string> v;
  vector<string> words;
  if (startString.size() > 0) {
    v.push_back(startString);
  }
  while (is) {
    line = getLine(is);
    words = getTokens(line);
    for (unsigned int i = 0; i < words.size(); ++i) {
      if (words[i][0] == '#') {
	break;
      }
      v.push_back(words[i]);
    }
  }
  argc = v.size();
  char **argv = 0;
  if (argc == 0) {
    return argv; // return NULL if no commands found
  }
  argv = (char **) malloc(argc * sizeof(char *));
  for (unsigned int i = 0; i < v.size(); ++i) {
    argv[i] = (char *) malloc((v[i].size() + 1) * sizeof(char) );
    for (unsigned int j = 0; j < v[i].size(); ++j) {
      argv[i][j] = v[i][j];
    }
    argv[i][v[i].size()] = '\0';
  }
  return argv;
}

// value = string:
void 
getArg( const char *givenOption, string &value, 
	int argc, char *argv[], string defaultValue)

{
  int countOption = 0, countValue = 0;
  string input;
  char option[80];

  bool justOption = false;
  option[0] = '\0';
  for( int i = 1; i < argc; i++ )
    {
      searchValueOfOption( option, input, argv[i] );
      // if there's an option, get this option:
      if ( argv[i][0] == '-' ) 
	{

	  // check if "option" is negative number
	  if ( (argv[i][1]<'0') || (argv[i][1]>'9') )
	    {
	      if( strcmp(givenOption, option) == 0 )
		{ 
		  countOption++; 
		}
	    }
 	} // if( argv[i][0] == '-' )
      // now interpret the input:
      if( input != "" )
	{
           if( strcmp(givenOption, option) == 0 )
		{ 
		  countValue++;
		  value = input;
		}
	} // if( input != "" )
    } // for( i < argc )

  // if an option is selected several times:
  if( countOption > 1 )
    {
      cerr << "You have given too often the option \"" 
	   << givenOption << "\"" << endl;
      ERROR( "Too many options", exception ); 
    }
  if( countValue > 1 )
    {
      cerr << "You have given too many input of the option \"" 
	   << givenOption << "\"" << endl;
      ERROR( "Too many options", exception ); 
    }
  if( justOption == false )
    { 
      if( (countOption==1) && (countValue==0) )
	{
	  cerr << "You have chosen the option \"" << givenOption 
	       << "\" but havn't specified the value!" << endl;
	  ERROR( "Missing value", exception ); 
	} // if( (countOption==1) && (countValue==0) &&)
      // check, if value is defined:
      if( value == "" )
	{ value = defaultValue; }
    } // if( justOption == false )
  else
    {
      if( (countOption==1) && (countValue==0) && (value=="") )
	{ value = defaultValue; }
      if( (countOption==0) && (countValue==0) )
	{ value = ""; }
    } // else (if(justOption==false))
}

void 
getArg( const char *givenOption, string &value, 
	int argc, char *argv[])
{
  string s = "";
  getArg(givenOption, value, argc, argv, s);
}

// value = integer:
void 
getArg( const char *givenOption, int &value, 
	     int argc, char *argv[], int defaultValue )
{
  string tempInt;
  
  value = defaultValue;
  getArg( givenOption, tempInt, argc, argv );

  if( tempInt != "" )
    { value = stoi( tempInt ); }
}

// value = integer:
void 
getArg( const char *givenOption, double &value, 
	     int argc, char *argv[], double defaultValue )
{
  string tempInt;
  
  value = defaultValue;
  getArg( givenOption, tempInt, argc, argv );

  if( tempInt != "" )
    { value = stod( tempInt ); }
}

/** reads matrix in format that is also used by program "R" */
Vec<Vec<double> >
readPlainMatrix(istream& is)
{
  Vec<Vec<double > > result;
  while (is) {
    string line = getLine(is);
    Vec<string> words = getTokens(line);
    if (words.size() > 0) {
      Vec<double> row(words.size());
      for (unsigned int i = 0; i < words.size(); ++i) {
	row[i] = stod(words[i]);
      }
      result.push_back(row);
    }
  }
  return result;
}

Vec<double>
readSequenceWeights(const string& fileName) {
  ifstream file(fileName.c_str());
  ERROR_IF(!file, "Error openening sequence weight file!", exception);
  int numEntries;
  file >> numEntries;
  ERROR_IF((!file) || (numEntries <= 0),
	   "Weight file: Number of entries expected as first item!",
	   exception);
  int id;
  Vec<double> result(numEntries, 0.0);
  for (int i = 0; i < numEntries; ++i) {
    file >> id >> result[i];
  }
  return result;
}

/** returns ACGT  or ACGU
 * TODO improve: consider protein alphabet etc
 */
string
estimateAlphabet(const Alignment& ali, const string alphabet)
{
  string result = "ACGU";
  if (ali.countCharacter('T') > ali.countCharacter('U')) {
    result = "ACGT";
  }
  return result;
}

/*
void
writeMatrix(ostream& os, const Vec<Vec<double> >& m) {
  for (unsigned int i = 0; i < m.size(); ++i) {
    for (unsigned int j = 0; j < m[i].size(); ++j) {
      os << m[i][j] << " ";
    }
    os << endl;
  }
}
*/

/** Computes mutual information and standard deviation of error correction for an alignment */
void
computeMatrices(Vec<Vec<double> >& matrix, 
		Vec<Vec<double> >& errorMatrix,
		Vec<double>& infVec,
		Vec<double>& infVecError,
		const Alignment& ali, 
		const MutualInformationScorer& matrixScorer)
{
  for (unsigned int i = 0; i < matrix.size(); ++i) {
    string s1 = ali.getSliceString(i);
    matrix[i][i] = 0.0;
    Physical infPhysical = matrixScorer.computeInformation(s1);
    infVec[i] = infPhysical.getValue();
    infVecError[i] = infPhysical.getError();
    for (unsigned int j = 0; j < i; ++j) {
      string s2 = ali.getSliceString(j);
      Physical physical = matrixScorer.computeMutualInformation(s1, s2);
      matrix[i][j] = physical.getValue();
      errorMatrix[i][j] = physical.getError();
      matrix[j][i] = matrix[i][j];
      errorMatrix[j][i] = errorMatrix[i][j];
    }
  }
}

/** returns length of c-style character array */
unsigned int
getCharLength(char * chars) {
  unsigned int charLen = 0;
  for (charLen = 0; chars[charLen] != '\0' ; ++charLen) {
  }
  return charLen;
}

/** returns zero if string and chars are the same.
 * Not using string.compare, because its signiture is
 * different for different g++ versions!!! :-(
 */
int
compareString(const string& s, 
	      char* chars,
	      string::size_type startPos,
	      string::size_type endPos) {
  unsigned int charLen = getCharLength(chars);
  if (startPos > charLen) {
    return 1;
  }
  for (string::size_type i = startPos; (i < s.size())&& (i<endPos); ++i) {
    if (chars[i] == '\0') {
      return 1;
    }
    else if (s[i] != chars[i]) {
      return -1;
    }
  }
  return 0;
}

/** returns string if environment variable set.
 * returns default value if env. variable not found.
 * TODO : cannot destinguish between a variable not set and set to empty string
 */
string
getEnv(const string& varName, char ** envp, string defaultValue) {
  string result = defaultValue;
  string::size_type compStart = 0; // first character to compare
  string::size_type varLength = varName.size();
  for (int i= 0; envp[i] != 0; i++) {
    char* line = envp[i];
    // if (varName.compare(line, compStart,varLength)==0) {
    if (compareString(varName, line, compStart,varLength)==0) {
      result = envp[i];
      // delete "+1" because of "=" sign
      result.erase(compStart,varLength+1);
      break;
    }
  }
  return result;
}

/** removes gap at position col of sequence with number collapseId 
    in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignment(Alignment& ali, 
		  unsigned int collapseId, 
		  unsigned int col)
{
  if ((collapseId >= ali.size()) || (col >= ali.getLength())
      || (ali.getLength() < 2)) {
    ERROR("Internal error in line 2584!", exception);
    return;
  }
  // deletePositionInStems(referenceStems, ali.getTemplate(collapseId), col); // order important!
  ali.deletePos(col);
}

/** returns true if character is gap character (. or -) */
bool
isGap(char c)
{
  return (c == '.') || (c == '-');
}

/** removes all gaps of sequence with number collapseId in alignment,
    adjusts alignment, probability matrix and reference stems. */
void
collapseAlignment(Alignment& ali, 
		  unsigned int collapseId)
{
  if (collapseId >= ali.size()) {
    return;
  }
  for (int i = ali.getLength()-1; i >= 0; --i) {
    if (isGap(ali.getTemplate(collapseId)[i])) {
      collapseAlignment(ali, collapseId, 
			static_cast<unsigned int>(i));
    }
  }
}

/** writes combined output format */
void
writeCombined(ostream& os, 
	      const Alignment& ali,
	      const Vec<Vec<double> >& matrix, 
	      const Vec<Vec<double> >& errorMatrix, 
	      const Vec<double>& infVec, 
	      const Vec<double>& infErrorVec, 
	      const Vec<Stem>& referenceStems,
	      double threshold, 
	      double relThreshold) 
{
  const string BR =  " <BR/>";
  os << "<HTML>" << endl;
  os << "<HEAD>" << endl;
  os << "<TITLE>CorreLogo results</TITLE>" << endl;
  os << "</HEAD>" << endl;
  os << "<BODY>" << endl;
  os << "### Values computed by CorreLogo version " << PROGRAM_VERSION << BR << endl;
  os << "### Information of individual alignment columns in bits per symbol: " << BR << endl;
  os << "### Alignment with " << ali.size() << " sequences and a length of " << ali.getLength() 
     << " nucleotides submitted." << BR << endl;
  unsigned int precision = 3;
  for (unsigned int i = 0; i < infVec.size(); ++i) {
    os << (i+1) << "\t" << setw(5+precision) << setprecision(precision) << infVec[i] << " +- " << setprecision(precision) <<  infErrorVec[i] << BR << endl;
  }
  os << "### Mutual Information of pairs of alignment columns in bits per symbol pair (absolute threshold: " 
     << threshold << " bits, relative threshold: " << relThreshold << " standard deviations) : " << BR << endl;
  for (unsigned int i = 0; i < infVec.size(); ++i) {
    for (unsigned int j = i + 1; j < infVec.size(); ++j) {
      if ((matrix[i][j] >= threshold) && ((errorMatrix[i][j] <= 0.0) || ((matrix[i][j]/errorMatrix[i][j]) >= relThreshold))) {
	os << (i+1) << " " << (j+1) << " \t" << setw(5+precision) 
	   << setprecision(precision) << matrix[i][j] << " +- " 
	   << setprecision(precision) << errorMatrix[i][j] << BR << endl;
      }
    }
  }
  os << "</BODY>" << endl;
  os << "</HEAD>" << endl;
}

double
computeResidueDistance(const Vec<Vector3D>& res1,
		       const Vec<Vector3D>& res2)
{
  double shortest = vecDistance(res1[0],res2[0]);
  double len;
  for (unsigned int i  = 0; i < res1.size(); ++i) {
    for (unsigned int j  = 0; j < res2.size(); ++j) {
      len = vecDistance(res1[i],res2[j]);
      if (len < shortest) {
	shortest = len;
      }
    }
  }
  return shortest;
}

/** workaround: generate stems of length 1 for each residue pair closer than cutoffDist */
Vec<Stem>
convertCoordinatesToContactStems(const Vec<Vec<Vector3D> >& pdbCoordinates, double cutoffDist) {
  Vec<Stem> result;
  for (unsigned int i = 0; i < pdbCoordinates.size(); ++i) {
    for (unsigned int j = i+2; j < pdbCoordinates.size(); ++j) {
      if (computeResidueDistance(pdbCoordinates[i], pdbCoordinates[j]) <= cutoffDist) {
	result.push_back(Stem(i,j, 1U));
      }
    }
  }
  return result;
}

int
main(int argc, char ** argv, char ** envp)
{
  bool readHydrogenMode = false;
  int argcFile = 0;
  int arrowMode = 1; // determines type of arrow being drawn (0: no arrow)
  int colorMode = 2;
  int collapseId = 0;
  int errorMode = 1;
  int gapMode = AlignColorConstants::LOGO_DEFAULT;
  int groundPlateMode = 0;
  int letterMode = Vrml2MatrixWriter::LETTER_MODE_TEXTURE;
  int letterMode2 = Vrml2MatrixWriter::LETTER_MODE_NONE;
  int logoMode = AlignColorConstants::LOGO_DEFAULT;
  int quitTERMode = 1;
  int readATOMMode = 1;
  int readHETATMMode = 1;
  int readingFrameMode = 0;
  // int zscoreMode = 0;
  int referenceFormat = 2;
  int reverseMatrixMode = 0;
  int textureMode = 1; // if zero, veto texture mapping
  int stemFormat = 1;
  char ** argvFile = 0;
  unsigned int pkCounter = 0;
  int verboseLevel = 0;
  // double gapLimit = 0.0;
  double cutoffDist = 4.0;
  double infLimit = 0.5;
  double infLimitRelative = 0;
  double infUpperLimit = -1.0;
  double matLimit = 0.5;
  // double stretchZ = 10.0;
  string alphabetString = "ACGU";
  string gapString = ".-";
  CharacterAlphabet alphabet(alphabetString, gapString); // defines allowed characters and gaps
  string bitMapFileName = "correlogo_bitmaps.dat";
  string comboMode;
  string commandFileName;
  string inputFileName;
  string logFileName; //  = "mainprogramtemplate.log";
  string matrixFileName;
  string outputFileNameOrig; //  = "aligncol.wrl";
  string pdbFileName;
  string resourceFileName;
  string referenceFileName;
  string rootDir = ".";
  string sequence;
  string sequenceType = "RNA";
  string stemFileName;
  string wwwImageBase; //  = "http://tetra.ncifcrf.gov:8080/rna/images";
  ResourceBundle resources;
  // Vrml1MatrixWriter vrml1Writer;
  Vrml2MatrixWriter vrml2Writer;
  // SimpleCompensationMatrixScorer matrixScorer;
  CorrectedMutualInformationScorer matrixScorer(alphabet);
  JVXMatrixWriter jvxWriter;
  Vec<int> outputFormats; // = VRML2_FORMAT;
  outputFormats.push_back(AlignColorConstants::JVX_FORMAT);
  outputFormats.push_back(AlignColorConstants::VRML2_FORMAT);
  outputFormats.push_back(AlignColorConstants::MATRIX_COMBINED_FORMAT);
  outputFormats.push_back(AlignColorConstants::MATRIX_FORMAT);
  int addedOutputFormat = -1;
  Vec<Vec<double> >  matrix;
  Vec<Vec<double> >  errorMatrix;
  Vec<double> stemScores;
  Vec<Letter3D> letters3D;
  Vec<Vec<Stem> > stemStack;
  Vec<Vec<Vector3D> > pdbCoordinates;

  Alignment ali;
  // CompensationScorer scorer, scorer2;
  Vec<Stem> referenceStems;

  if (argc < 2)  {
    helpOutput(cout);
    exit(0);
  }
  getArg("-verbose", verboseLevel, argcFile, argvFile, verboseLevel);
  getArg("-verbose", verboseLevel, argc, argv, verboseLevel);
  const string envrootDirVarName = "CORRELOGO_HOME";
  string envRootDir;
  envRootDir = getEnv(envrootDirVarName, envp, envRootDir);
  if (envRootDir.size() == 0) {
    cout << "Warning: variable "
	 << envrootDirVarName << " not properly set!" << endl;
  } 
  else {
    rootDir = envRootDir + SLASH_STR + "resources";
  }
  getArg("-root", rootDir, argc, argv, rootDir);
  if (verboseLevel > 1) {
    cout << "Using root directory: " << rootDir << endl;
  }
  addSlash(rootDir);

  getArg("-commands", commandFileName, argc, argv, commandFileName);
  addPathIfRelative(commandFileName, rootDir);

  if (commandFileName.size() > 0) {
    ifstream commandFile(commandFileName.c_str());
    if (!commandFile) {
      if (isPresent("-commands", argc, argv)) {
	ERROR_IF(!commandFile, "Error opening command file.", exception);
      }
      else {
	cerr << "Warning: Could not find command file: " + commandFileName 
	     << endl;
      }
    }
    else {
      argvFile = streamToCommands(commandFile, argcFile, 
				  string("mainprogramtemplate"));
    }
    commandFile.close();
  }

  getArg("-arrow", arrowMode, argcFile, argvFile, arrowMode);
  getArg("-arrow", arrowMode, argc, argv, arrowMode);
  getArg("-bitmaps", bitMapFileName, argcFile, argvFile, bitMapFileName);
  getArg("-bitmaps", bitMapFileName, argc, argv, bitMapFileName);
  getArg("b", bitMapFileName, argcFile, argvFile, bitMapFileName);
  getArg("b", bitMapFileName, argc, argv, bitMapFileName);
  // cout << "Supplied bitmap file: " << bitMapFileName << endl;
  addPathIfRelative(bitMapFileName, rootDir);  
  // cout << "Used bitmap file: " << bitMapFileName << endl;
  getArg("c", colorMode, argcFile, argvFile, colorMode);
  getArg("c", colorMode, argc, argv, colorMode);
  getArg("-collapse", collapseId, argcFile, argvFile, collapseId);
  getArg("-collapse", collapseId, argc, argv, collapseId);
  --collapseId; // internal counting starts at zero
  getArg("-combo", comboMode, argcFile, argvFile, comboMode);
  getArg("-combo", comboMode, argc, argv, comboMode);
  getArg("d", cutoffDist, argcFile, argvFile, cutoffDist);
  getArg("d", cutoffDist, argc, argv, cutoffDist);
  getArg("-error", errorMode, argcFile, argvFile, errorMode);
  getArg("-error", errorMode, argc, argv, errorMode);
  getArg("-gaps", gapMode, argcFile, argvFile, gapMode);
  getArg("-gaps", gapMode, argc, argv, gapMode);
  getArg("-ground", groundPlateMode, argcFile, argvFile, groundPlateMode);
  getArg("-ground", groundPlateMode, argc, argv, groundPlateMode);
  getArg("i", inputFileName, argcFile, argvFile, inputFileName);
  getArg("i", inputFileName, argc, argv, inputFileName);
  getArg("r", referenceFileName, argc, argv, referenceFileName);
  if (referenceFileName.compare("null") == 0 ) {
    referenceFileName = ""; // trick for allowing non-specified filenames
  }
  getArg("-rf", referenceFormat, argc, argv, referenceFormat);
  // getArg("l", infLimit, argcFile, argvFile, infLimit);
  getArg("l", infLimit, argc, argv, infLimit);
  //   getArg("-lr", infLimitRelative, argcFile, argvFile, infLimitRelative);
  getArg("-lr", infLimitRelative, argc, argv, infLimitRelative);
  getArg("-lu", infUpperLimit, argc, argv, infUpperLimit);
  getArg("-lm", matLimit, argc, argv, matLimit);
  getArg("-letter", letterMode, argcFile, argvFile, letterMode);
  getArg("-letter", letterMode, argc, argv, letterMode);
  getArg("-letter2", letterMode2, argcFile, argvFile, letterMode2);
  getArg("-letter2", letterMode2, argc, argv, letterMode2);
  getArg("-logo", logoMode, argcFile, argvFile, logoMode);
  getArg("-logo", logoMode, argc, argv, logoMode);
  getArg("m", matrixFileName, argc, argv, matrixFileName);
  getArg("-log", logFileName, argc, argv, logFileName);
  getArg("-log", logFileName, argcFile, argvFile, logFileName);
  addPathIfRelative(logFileName, rootDir);
  getArg("o", outputFileNameOrig, argcFile, argvFile, outputFileNameOrig);
  getArg("o", outputFileNameOrig, argc, argv, outputFileNameOrig);
  getArg("-of", addedOutputFormat, argc, argv, addedOutputFormat);
  if (addedOutputFormat >  0) {
    outputFormats.push_back(addedOutputFormat);
  }
  getArg("-stems", stemFileName, argcFile, argvFile, stemFileName);
  getArg("-stems", stemFileName, argc, argv, stemFileName);
  getArg("-quit-ter", quitTERMode, argcFile, argvFile, quitTERMode);
  getArg("-quit-ter", quitTERMode, argc, argv, quitTERMode);
  getArg("p", resourceFileName, argcFile, argvFile, resourceFileName);
  getArg("p", resourceFileName, argc, argv, resourceFileName);
  getArg("-pdb", pdbFileName, argc, argv, pdbFileName);
  if (pdbFileName.compare("null") == 0 ) {
    pdbFileName = ""; // trick for allowing non-specified filenames
  }
  getArg("-read-atom", readATOMMode, argcFile, argvFile, readATOMMode);
  getArg("-read-atom", readATOMMode, argc, argv, readATOMMode);
  getArg("-read-hetatm", readHETATMMode, argcFile, argvFile, readHETATMMode);
  getArg("-read-hetatm", readHETATMMode, argc, argv, readHETATMMode);
  getArg("-reading-frame", readingFrameMode, argcFile, argvFile, readingFrameMode);
  getArg("-reading-frame", readingFrameMode, argc, argv, readingFrameMode);
  getArg("-reverse", reverseMatrixMode, argcFile, argvFile, reverseMatrixMode);
  getArg("-reverse", reverseMatrixMode, argc, argv, reverseMatrixMode);
  getArg("-texture", textureMode, argcFile, argvFile, textureMode);
  getArg("-texture", textureMode, argc, argv, textureMode);
  getArg("-type", sequenceType, argcFile, argvFile, sequenceType);
  getArg("-type", sequenceType, argc, argv, sequenceType);
  getArg("-www", wwwImageBase, argcFile, argvFile, wwwImageBase);
  getArg("-www", wwwImageBase, argc, argv, wwwImageBase);
  if (logFileName.size() > 0) {
    ofstream logFile(logFileName.c_str(), ios::app);
    parameterOutput(logFile, argc, argv);
    if (argcFile > 1) {
      logFile << "Parameters from command file: ";
      parameterOutput(logFile, argcFile, argvFile);
    }
    logFile.close();
  }

  /***************** MAIN PROGRAM *****************************/

  cout << "# correlogo version " << PROGRAM_VERSION << " called with parameters: ";
  parameterOutput(cout, argc, argv);

  if (comboMode.compare("3d") == 0) {
    cout << "Optimizing output for 3D printing!" << endl;
    arrowMode = 0;
    gapMode = AlignColorConstants::LOGO_NONE;
    groundPlateMode = 1;
    logoMode = AlignColorConstants::LOGO_NONE;
  }

  if (resourceFileName.size() > 0) {
    ifstream resourceFile(resourceFileName.c_str());
    ERROR_IF(!resourceFile, "Error opening resource file!",
	     exception);
    resourceFile >> resources;
    vrml2Writer.setResourceBundle(resources);
    // vrml1Writer.setResourceBundle(resources);
    jvxWriter.setResourceBundle(resources);
    cout << "Read resources: " << endl << resources << endl;
    resourceFile.close();
  }

  if (bitMapFileName.size() > 0) {
    cout << "Reading parameter file: " << bitMapFileName << endl;
    ifstream bitMapFile(bitMapFileName.c_str());
    ERROR_IF(!bitMapFile, "Error reading parameter file!", exception);
    bitMapFile >> letters3D;
    if (!bitMapFile) {
      ERROR("Error reading bitmap file!", exception);
    }
    jvxWriter.setLetters3D(letters3D);
    vrml2Writer.setLetters3D(letters3D);
    bitMapFile.close();
  }

  if (groundPlateMode) {
    jvxWriter.setGroundPlateMode(true);
    vrml2Writer.setGroundPlateMode(true);
  }

  vrml2Writer.setArrowMode(arrowMode);
  vrml2Writer.setColorMode(colorMode);
  vrml2Writer.setErrorMode(errorMode);
  vrml2Writer.setGapMode(gapMode);
  vrml2Writer.setInfLimit(infLimit);
  vrml2Writer.setInfLimitRelative(infLimitRelative);
  vrml2Writer.setInfUpperLimit(infUpperLimit);
  vrml2Writer.setLetterMode(letterMode);
  vrml2Writer.setLetterMode2(letterMode2);
  vrml2Writer.setLogoMode(logoMode);
  vrml2Writer.setReadingFrameMode(readingFrameMode);
  vrml2Writer.setSequenceType(sequenceType);
  jvxWriter.setArrowMode(arrowMode);
  jvxWriter.setColorMode(colorMode);
  jvxWriter.setErrorMode(errorMode);
  jvxWriter.setGapMode(gapMode);
  jvxWriter.setInfLimit(infLimit);
  jvxWriter.setInfLimitRelative(infLimitRelative);
  jvxWriter.setInfUpperLimit(infUpperLimit);
  jvxWriter.setLetterMode(letterMode);
  jvxWriter.setLetterMode2(letterMode2);
  jvxWriter.setLogoMode(logoMode);
  jvxWriter.setReadingFrameMode(readingFrameMode);
  jvxWriter.setSequenceType(sequenceType);
  cout << "The obtained sequence type is: " << sequenceType << endl;
  if (sequenceType.compare("DNA") == 0) {
    cout << "DNA sequence recognized!" << endl;
    vrml2Writer.setAlphabet(DNA_ALPHABET);
    jvxWriter.setAlphabet(DNA_ALPHABET);
  }
  else if (sequenceType.compare("RNA") == 0) {
    cout << "RNA sequence recognized!" << endl;
    vrml2Writer.setAlphabet(RNA_ALPHABET);
    jvxWriter.setAlphabet(RNA_ALPHABET);
  }
  else {
    ERROR("Sequence type unrecognized!", exception);
  }
  // cout << "Defined scorer alphabets: " << scorer.getAlphabet() << " " << scorer2.getAlphabet() << endl;
  if (isPresent("-www", argc, argv)) {
    // vrml1Writer.setWwwImageBase(wwwImageBase);
    vrml2Writer.setWwwImageBase(wwwImageBase);
    jvxWriter.setWwwImageBase(wwwImageBase);
  }
  if (textureMode == 0) {
    // vrml1Writer.setWwwImageBase(string(""));
    vrml2Writer.setWwwImageBase(string(""));
    jvxWriter.setWwwImageBase(string(""));
  }

  if (referenceFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading reference stems from: " << referenceFileName << endl;
    }
    ifstream stemFile(referenceFileName.c_str());
    ERROR_IF(!stemFile, "Error opening reference stem file!", exception);
    switch (referenceFormat) {
    case 1:
      // region file
      referenceStems = readStems(stemFile);
      break;
    case 2:
      // bracket notation
      referenceStems = stemsFromBracketFasta(stemFile, pkCounter);
      break;
    default:
      ERROR("Unknown stem file format!", exception);
    }
    stemFile.close();
    ERROR_IF(referenceStems.size() == 0,
	     "No reference stems read!", exception);
    if (verboseLevel > 0) {
      cout << "Read reference stems: " << endl;
      writeStems(cout, referenceStems);
    }
  }

  if (stemFileName.size() > 0) {
    if (verboseLevel > 0) {
      cout << "Reading stems from : " << stemFileName.size() << endl;
    }
    ifstream stemFile(stemFileName.c_str());
    ERROR_IF(!stemFile, "Error opening stem file!", exception);
    switch (stemFormat) {
    case 1:
      // region file
      stemStack = readMultiStems(stemFile);
      break;
    case 2:
      // bracket notation
      stemStack = readRNAsubopt(stemFile, stemScores);
      break;
    case 3:
      stemStack = readCtCombined(stemFile, sequence, 1);
      break;
    default:
      ERROR("Unknown stem file format!", exception);
    }
  }

  if (inputFileName.size() > 0) {
    ifstream inputFile(inputFileName.c_str());
    ERROR_IF(!inputFile, "Error opening input file!", exception);    
    ali.loadFasta(inputFile);    
    inputFile.close();
    cout << "Alignment with " << ali.size() << " sequences of length " 
	 << ali.getLength() << " read." << endl;
    // adjust sequences:
    cout << "Adjusting to capitol letters and possibly replacing T and U in alignment" 
	 << endl;
    ali.upperCaseSequences();
    if (sequenceType.compare("RNA") == 0) {
      ali.replaceChar('T', 'U'); // replace DNA alphabet to RNA
    }
    else if (sequenceType.compare("DNA") == 0) {
      ali.replaceChar('U', 'T'); // replace DNA alphabet to RNA
    }
    alphabetString = estimateAlphabet(ali, alphabetString);
    alphabet = CharacterAlphabet(alphabetString, gapString); // possible use ACGT alphabet
    if (verboseLevel > 1) {
      cout << "Read fasta file: " << endl;
      ali.saveFasta(cout);
    }
  }

  if (collapseId >= 0) {
    if (verboseLevel > 0) {
      cout << "Collapsing gaps with respect to sequence "
	   << collapseId + 1 << " " << ali.getTemplateName(collapseId) << endl;
    }
    collapseAlignment(ali, (unsigned int)collapseId);
    if (verboseLevel > 0) {
      cout << "New length of alignment after collapsing: " << ali.getLength() << endl;
    }
  }

  if (pdbFileName.size() > 0) {
    ifstream pdbFile(pdbFileName.c_str());
    ERROR_IF(!pdbFile, "Error opening pdb file!", exception);
    pdbCoordinates = readPdbVectors(pdbFile, readHydrogenMode,
				    readATOMMode, readHETATMMode, quitTERMode);
    if (verboseLevel > 0) {
      cout << "PDB file with " << pdbCoordinates.size() << " residues read."
	   << endl;
    }
    if (verboseLevel > 1) {
      cout << "Read pdb coordinates: " << endl << pdbCoordinates << endl;
    }
    ERROR_IF(pdbCoordinates.size() != ali.getLength(), 
	     "Number of residues in PDB file does not match number of residues in alignment!", exception); 
  }
  if (pdbCoordinates.size() > 0) {
    // workaround: generate stems of length 1 for each residue pair closer than cutoffDist
    referenceStems = convertCoordinatesToContactStems(pdbCoordinates, cutoffDist);
    if (verboseLevel > 0) {
      cout << "Generated PDB contacts: " << referenceStems << endl;
    }
  }

  if (matrixFileName.size() > 0) {
    ifstream inputFile(matrixFileName.c_str());
    ERROR_IF(!inputFile, "Error opening input file!", exception);
    matrix = readPlainMatrix(inputFile);
    if (reverseMatrixMode) {
      reverse(matrix.begin(), matrix.end());
    }
    inputFile.close();    
    cout << matrix.size() << " by " << matrix[0].size() << " matrix read" 
	 << endl;
  }
  else {
    matrix = Vec<Vec<double> >(ali.getLength(), 
			       Vec<double>(ali.getLength(), 0.0));
  }

  // output of 3D models:

  errorMatrix = Vec<Vec<double> >(matrix.size(), Vec<double>(matrix.size(), 0.0));
  // height of regular 2d sequence logos is stored here:
  Vec<double> infVec(matrix.size(), 0.0);
  Vec<double> infErrorVec(matrix.size(), 0.0);
  // compute matrix if needed: 
  if (matrixFileName.size() == 0) {
    computeMatrices(matrix, errorMatrix, 
		    infVec, infErrorVec,
		    ali, matrixScorer);
  }
  if (verboseLevel > 2) {
    cout << "the raw matrix is: " << endl << matrix << endl;
    cout << "the error matrix is: " << endl << errorMatrix << endl;
  }

  for (Vec<int>::size_type formatIdx = 0; formatIdx < outputFormats.size(); ++formatIdx) {
    int outputFormat = outputFormats[formatIdx];
    string outputFileName;
    if (outputFileNameOrig.size() == 0) {
      outputFileName = fileNameBase(inputFileName);
    }
    else {
      outputFileName = outputFileNameOrig;
    }
    switch (outputFormat) {
    case AlignColorConstants::JVX_FORMAT:
      outputFileName += ".jvx";
      break;
    case AlignColorConstants::VRML2_FORMAT:
      outputFileName += ".wrl";
      break;
    case AlignColorConstants::MATRIX_FORMAT:
      outputFileName += ".matrix";
      break;
    case AlignColorConstants::ERROR_MATRIX_FORMAT:
      outputFileName += "_error.matrix";
      break;
    case AlignColorConstants::MATRIX_COMBINED_FORMAT:
      outputFileName += ".txt";
      break;
    default:
      ERROR("Unknown output format id!", exception);
    }
    cout << "Writing output file: " << outputFileName << endl;
    ofstream outputFile(outputFileName.c_str());
    ERROR_IF(!outputFile, "Error opening output file!", exception);
    // cout << "Writing output to file: " << outputFileName << endl;
    if (ali.size() > 0) {
      switch (outputFormat) {
      case AlignColorConstants::VRML2_FORMAT: 
	vrml2Writer.writeVrml(outputFile, ali, matrix, errorMatrix, 
			      infVec, infErrorVec, referenceStems);
	break;
      case AlignColorConstants::JVX_FORMAT:
	jvxWriter.writeVrml(outputFile, ali, matrix, errorMatrix, 
			    infVec, infErrorVec, referenceStems);
	break;
      case AlignColorConstants::MATRIX_FORMAT:
	writeMatrix(outputFile, matrix);
	break;
      case AlignColorConstants::ERROR_MATRIX_FORMAT:
	writeMatrix(outputFile, errorMatrix);
	break;
      case AlignColorConstants::MATRIX_COMBINED_FORMAT:
	writeCombined(outputFile, ali, matrix, errorMatrix, infVec, infErrorVec, referenceStems, infLimit, infLimitRelative);
	break;
      default:
	ERROR("Unknown output format id!", exception);
      }
    }
    else {
      ERROR("No alignment defined!", exception);
    }
    outputFile.close();
  }

  return 0;
}
